///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////
package com.crankuptheamps.client;

import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.fields.BookmarkField;
import com.crankuptheamps.client.fields.Field;
/**
 * A ring buffer of bookmarks and activation status
 * Used by all of the bookmark stores to track state of bookmarks
 * we need to hold on to, either because they're active or because
 * they're after an active one.
 *
 */
public class BookmarkRingBuffer
{
    /**
     * Represents a single entry in an array of bookmarks
     *
     *
     */
    public class Entry
    {
        BookmarkField _bookmark;
        long _index = 0;
        boolean _active = false;
        boolean _persisted = false;

        public Entry()
        {
            _bookmark = new BookmarkField();
        }
        public Entry setBookmark(Field f)
        {
            _bookmark.copyFrom(f);
            return this;
        }
        public BookmarkField getBookmark()
        {
            return _bookmark;
        }
        public Entry setIndex(long index)
        {
            _index = index;
            return this;
        }
        public long getIndex()
        {
            return _index;
        }
        public boolean isActive()
        {
            return _active;
        }
        public Entry setActive(boolean active)
        {
            _active = active;
            return this;
        }
        public boolean isPersisted()
        {
            return _persisted;
        }
        public Entry setPersisted(boolean persisted)
        {
            _persisted = persisted;
            return this;
        }

    }
    final int INITIAL_ENTRY_COUNT = 1000;
    public final static int UNSET_INDEX = -1;
    Entry _entries[] = new Entry[INITIAL_ENTRY_COUNT];
    int _start = 0, _end = 0;
    int _recovery = UNSET_INDEX, _recoveryMax = UNSET_INDEX;
    long _index = 0, _indexOfStart = 0, _indexOfRecovery = UNSET_INDEX;
    BookmarkField _lastDiscarded = new BookmarkField();
    boolean _persistedAcks = true, _empty = true;
    BookmarkStoreResizeHandler _resizeHandler;
    BookmarkStore _store;
    Field _subId;

    /**
     * Initializes the underlying array, and sets
     * the "last discarded" value to something reasonable.
     */
    public BookmarkRingBuffer()
    {
        _lastDiscarded.set("0".getBytes());
        for(int i = 0; i < _entries.length; ++i)
        {
            _entries[i] = new Entry();
        }
    }

    /**
     * Size of underlying array
     * @return current capacity of self.
     */
    public int capacity()
    {
        return _entries.length;
    }

    /**
     * Retrieves an Entry given an index
     * @param index An index returned by .getIndex(), getStartIndex(),
     *              getEndIndex(), log, etc.
     * @return The entry at that index, or null if not found.
     */
    public Entry getByIndex(long index)
    {
        // The index must be higher than the lowest we have stored, which is
        // either _start+_indexOfStart or _recovery+_indexOfRecovery if we
        // are still recovering.
        int offsetFromStart = 0;
        if (_recovery == UNSET_INDEX)
        {
            if (index < _indexOfStart + _start)
                return null;
            offsetFromStart = (int)(index-_indexOfStart) % _entries.length;
        }
        else
        {
            if (index < _recovery + _indexOfRecovery)
                return null;
            offsetFromStart = (int)(index-_indexOfRecovery) % _entries.length;
        }
        return _entries[offsetFromStart];
    }

    /**
     * Returns the "last discarded" bookmark. This may be a
     * newer bookmark than the one you last passed to discard(),
     * if you're discarding out of order.
     *
     * @return the last bookmark to be discarded.
     */
    public BookmarkField getLastDiscarded()
    {
        return _lastDiscarded;
    }

    /**
     * Returns if the buffer is currently empty.
     * @return true if self is empty and false otherwise.
     */
    public boolean isEmpty()
    {
        return _empty;
    }

    /**
     * Returns the index value associated with the first valid
     * Entry in self.
     * @return index of first valid entry.
     */
    public long getStartIndex()
    {
        if (_empty) return UNSET_INDEX;
        return _indexOfStart + _start;
    }

    /**
     * Returns the index value one greater than the last valid
     * Entry in self.
     * @return index 1 greater than last valid entry.
     */
    public long getEndIndex()
    {
        if (_empty) return UNSET_INDEX;
        return _index;
    }

    /**
     * Logs the bookmark by allocating an Entry, setting
     * the Entry to active, copying the bookmark value
     * to that entry, and returning the index
     * of that entry.
     * @param bookmark the bookmark to loc
     * @return the index of the new Entry.
     */
    public long log(BookmarkField bookmark)
    {
        _empty = false;
        int writeIndex = _end++;
        _entries[writeIndex].setBookmark(bookmark).setActive(true).setPersisted(!_persistedAcks);
        if(_end == _entries.length) _end = 0;
        int min = (_recovery == UNSET_INDEX) ? _start : _recovery;
        while(_end == min)
        {
            resize();
            min = (_recovery == UNSET_INDEX) ? _start : _recovery;
        }
        return _index++;
    }

    /**
     * Re-logs the bookmark by copying an Entry to a new location
     * and returning the index
     * of that new location.
     * @param bookmark the index to move
     * @return the index of the new Entry.
     */
    public long relog(long oldIndex, BookmarkField bookmark)
    {
        _empty = false;
        int writeIndex = _end++;
        // Recovery could have been reset because remaining recovered bookmarks
        // were persisted, but in some cases they could get relogged
        long baseIndex = (_indexOfRecovery == UNSET_INDEX) ? _indexOfStart
                                                           : _indexOfRecovery;
        // Also possible that this bookmark is at the tail of the buffer and
        // recovery was reset to start of buffer causing baseIndex to be > oldIndex
        if (oldIndex < baseIndex) oldIndex += _entries.length;
        int offsetFromStart = (int)(oldIndex - baseIndex) % _entries.length;
        Entry current = _entries[offsetFromStart];
        _entries[writeIndex].setBookmark(bookmark).setActive(current.isActive()).setPersisted(current.isPersisted() || !_persistedAcks);

        // New one is only version, so mark the old one as complete
        current.setActive(false).setPersisted(true).getBookmark().reset();
        resetRecovery();
        if(_end == _entries.length) _end = 0;
        int min = (_recovery == UNSET_INDEX) ? _start : _recovery;
        while(_end == min)
        {
            resize();
            min = (_recovery == UNSET_INDEX) ? _start : _recovery;
        }
        return _index++;
    }

    /**
     * Discards an entry by index. If the discard is completed,
     * lastDiscarded will change, otherwise the discard is cached
     * in the entry, and the getLastDiscarded() value is unchanged.
     *
     * @param index the index of the entry.
     * @return true if this caused lastDiscarded to change.
     */
    public boolean discard(long index)
    {
        long minIndex = 0;
        if (_recovery == UNSET_INDEX)
        {
            minIndex = _indexOfStart;
        }
        else
        {
            minIndex = _indexOfRecovery;
        }

        if (index < minIndex || index > _index)
        {
            return false;
        }

        int offsetFromStart = (int)(index - minIndex);
        int indexInArray = offsetFromStart % _entries.length;

        _entries[indexInArray].setActive(false);
        if(_start == indexInArray)
        {
            Entry lastDiscardedEntry = null;
            while(!_entries[_start].isActive() && _entries[_start].isPersisted() &&
                  (_start != _end || !_empty))
            {
                if (lastDiscardedEntry != null)
                {
                    lastDiscardedEntry.getBookmark().reset();
                }
                lastDiscardedEntry = _entries[_start];
                if(++_start == _entries.length)
                {
                    _start = 0;
                    _indexOfStart += _entries.length;
                }
                _empty = (_start == _end);
            }
            if(lastDiscardedEntry != null)
            {
                _lastDiscarded.copyFrom(lastDiscardedEntry.getBookmark());
                lastDiscardedEntry.getBookmark().reset();
                return true;
            }
        }
        else if (_recovery == indexInArray)
        {
            Entry lastDiscardedEntry = resetRecovery();
            if(lastDiscardedEntry != null)
            {
                _lastDiscarded.copyFrom(lastDiscardedEntry.getBookmark());
                lastDiscardedEntry.getBookmark().reset();
                return true;
            }
        }
        return false;
    }

    private Entry resetRecovery()
    {
        if (_recovery == UNSET_INDEX)
            return null;
        Entry lastDiscardedEntry = null;
        while ( !_entries[_recovery].isActive() && _entries[_recovery].isPersisted())
        {
            if (_start == _end)
            {
                if (lastDiscardedEntry != null)
                {
                    lastDiscardedEntry.getBookmark().reset();
                }
                lastDiscardedEntry = _entries[_recovery];
            }
            if (_recovery == _recoveryMax)
            {
                _recovery = UNSET_INDEX;
                _indexOfRecovery = UNSET_INDEX;
                _recoveryMax = UNSET_INDEX;
                _empty = (_start == _end);
                break;
            }
            else if (++_recovery == _entries.length)
            {
                _recovery = 0;
                _indexOfRecovery += _entries.length;
            }
        }
        return lastDiscardedEntry;
    }

    /**
     * Searches valid Entrys for the given bookmark
     * @param field the bookmark to search for
     * @return the Entry containing the given bookmark, or null.
     */
    public Entry find(BookmarkField field)
    {
        if (_empty) return null;
        long index = _indexOfStart + _start;
        int i = _start;
        do
        {
            if(!_entries[i].getBookmark().isNull() &&
               _entries[i].getBookmark().equals(field))
            {
                // hooray! we found it.
                return _entries[i].setIndex(index);
            }
            ++index;
            i = (i+1)%_entries.length;
        } while(i!=_end);
        return null;
    }

    /**
     * Mark all records up to the and including the provided bookmark as safe for discard and discard all appropriate bookmarks. Applies to server versions 3.8x.
     * @param bookmark The latest bookmark that can be safely disposed.
     */
    public void persisted(BookmarkField bookmark)
    {
        Entry safeEntry = find(bookmark);
        if (safeEntry == null)
        {
            return;
        }
        long last = safeEntry.getIndex();
        persisted(last);
    }

    public void persisted(long bookmark)
    {
        // Make sure this bookmark is in current valid range
        if (bookmark < getStartIndex() || bookmark > getEndIndex()) return;
        boolean foundActive = false;
        Entry lastDiscardedEntry = null;
        long startIndex = _indexOfStart;
        // Mark all bookmarks up to the provided one as persisted
        // Keep track of lastDiscardedEntry as we go through so that
        // most recent can be updated. Stop tracking as soon as we find
        // a bookmark that is still active (not discarded)
        for (int idx = _start; idx+startIndex <= bookmark; )
        {
            _entries[idx].setPersisted(true);
            if (!foundActive)
            {
                if (!_entries[idx].isActive() && !_entries[idx].getBookmark().isNull())
                {
                    // We've found a new most recent, clear previous and point to new
                    if (lastDiscardedEntry != null)
                    {
                        lastDiscardedEntry.getBookmark().reset();
                    }
                    lastDiscardedEntry = _entries[idx];
                    // Advance start since this one is now complete
                    ++_start;
                }
                else
                {
                    // Found an active bookmark, so no more removal.
                    foundActive = true;
                }
            }
            ++idx;
            // Reached the buffer end, circle back to start of buffer
            if (idx == _entries.length)
            {
                // If we're still updating recent, also reset start
                if (_start == idx)
                {
                    _start = 0;
                    _indexOfStart += _entries.length;
                }
                startIndex += _entries.length;
                idx = 0;
            }
        }
        // We've got a new mostRecent bookmark, so update
        if (lastDiscardedEntry != null)
        {
            _lastDiscarded.copyFrom(lastDiscardedEntry.getBookmark());
            lastDiscardedEntry.getBookmark().reset();
        }
        // Flag if we're empty now
        _empty = (_start == _end);
    }

    /**
     * Called to indicate that the server won't provide persisted acks for the bookmarks
     * so anything that is discarded should be assumed persisted.
     */
    public void noPersistedAcks()
    {
        _persistedAcks = false;
        persisted(_index);
    }

    /**
     * Called to indicate that the server will provide persisted acks for the bookmarks.
     */
    public void setPersistedAcks()
    {
        _persistedAcks = true;
    }

    /**
     * Called to check if persisted acks are expected.
     */
    public boolean persistedAcks()
    {
        return _persistedAcks;
    }

    /**
     * Called when self is full, assumes all entries are valid.
     */
    private void resize()
    {
        int oldLength = _entries.length;
        if (_resizeHandler != null && !_resizeHandler.invoke(_store, _subId, (int)(oldLength*1.5)))
        {
            return;
        }
        // allocate a new array (1.5x resize factor)
        // copy over the data.
        Entry[] newEntries = new Entry[(int) (oldLength * 1.5)];

        int start = ((_recovery == UNSET_INDEX) ? _start : _recovery);
        System.arraycopy(_entries, start, newEntries, 0, oldLength - start);
        if(start > 0)
        {
            System.arraycopy(_entries, 0, newEntries, oldLength - start, start);
        }
        if (_recovery != UNSET_INDEX)
        {
            _indexOfStart += _recovery;
            _indexOfRecovery += _recovery;
            _start = _start - _recovery;
            if (_start<0) _start += _entries.length;
            _recoveryMax = _recoveryMax - _recovery;
            _recovery = 0;
        }
        else
        {
            _indexOfStart += _start;
            _start = 0;
        }
        _end = oldLength;
        for(int i = oldLength; i< newEntries.length; ++i)
        {
            newEntries[i] = new Entry();
        }
        _entries = newEntries;
    }

    public void setResizeHandler(BookmarkStoreResizeHandler handler, BookmarkStore store)
    {
        _resizeHandler = handler;
        _store = store;
    }

    public void setSubId(Field sub)
    {
        _subId = sub.copy();
    }

    public long setRecovery()
    {
        if (_recovery == UNSET_INDEX)
        {
            if (_empty) return UNSET_INDEX;
            _recovery = _start;
            _indexOfRecovery = _indexOfStart;
        }
        _recoveryMax = _end - 1;
        if (_recoveryMax < 0) _recoveryMax = _entries.length - 1;
        if (_end < _start) _indexOfStart += _entries.length;
        _start = _end;
        return _recovery + _indexOfRecovery;
    }

}
