////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.spark;

import java.util.UUID;

import com.crankuptheamps.client.Client;
import com.crankuptheamps.client.ClientDisconnectHandler;
import com.crankuptheamps.client.CommandId;
import com.crankuptheamps.client.Message;
import com.crankuptheamps.client.MessageHandler;
import com.crankuptheamps.client.exception.ConnectionException;


public class SOWAndSubscribeCommand extends SparkCommand
{
    public SOWAndSubscribeCommand(String name)
    {
        super(name);

        // These are the arguments this command recognizes
        addOption(new CommandOption("topic", "topic to query", true));
        addOption(new CommandOption("filter", "content filter to use", false));
        addOption(new CommandOption("batchsize", "batch size to use during query", false));
        addOption(new CommandOption("delta", "request deltas for the subscription", false));
        addOption(new CommandOption("bookmark", "bookmark for historical sow query", false));
        addOption(new CommandOption("orderby", "orderby expression for sow query", false));
        addOption(new CommandOption("topn", "only return top n records for sow query", false));
        addOption(new CommandOption("opts", "additional options: comma separated list of [oof,live,send_keys,no_empties]", false));
        addOption(new CommandOption("timeout", "timeout to wait for command to return in milliseconds", false));
        
        // These are the examples...
        StringBuilder example = new StringBuilder(1024);
        example
        .append("  % ./spark sow_and_subscribe -server localhost:9003 -topic Trades -delta -filter \"/55 = 'Transport.123'\"\n");
        example.append("    Connects to the AMPS instance listening on port 9003");
        example.append("    and prints records in topic 'Trades' that have");
        example.append("    tag 55 set to 'Transport.123' and then return message");
        example.append("    deltas over the subscription.");
        addExample(example.toString());

    }

    private static class SOWAndSubscribeCommandHandler implements MessageHandler, ClientDisconnectHandler
    {
        public void invoke(Message message)
        {
            switch (message.getCommand())
            {
            case Message.Command.GroupBegin:
                // The GroupBegin comes just before the query results
                break;
            case Message.Command.SOW:
                // Here's the SOW record data
                System.out.println(message.getData());
                break;
            case Message.Command.Publish:
                // Here's the publish record (comes after the SOW results)
                System.out.println(message.getData());
                break;
            case Message.Command.OOF:
                // Here's the oof record (comes after the SOW results)
                break;
            case Message.Command.GroupEnd:
                // Once we see GroupEnd, it means we've seen the entire result
                break;
            }
        }
        public void invoke(Client client)
        {
            // We've been disconnected, let's just quit.
            System.err.println("Disconnected");
            synchronized(this)
            {
                this.notifyAll();
            }
        }
    }

    void execute(String argv[])
    {
        // Extract all of the command line arguments
        parseArgs(argv, 1);
        String[] userAndServer = options.get("server").getValue().split("@",2);
        String server = (userAndServer.length == 2) ? userAndServer[1] : userAndServer[0];
        String topic   = options.get("topic").getValue();
        String type    = options.get("proto").getValue();
        boolean delta  = options.get("delta").hasValue();
        String filter  = null;
        String bookmark  = null;
        String orderBy  = null;
        String opts  = null;
        int topN  = 0;
        int batchSize  = 1;
        int timeout = 5000;

        if(!type.equals("xml") && !type.equals("fix") && !type.equals("nvfix") && !type.equals("amps") && !type.equals("json"))
        {
            badUsage("prot argument needs to be 'fix', 'xml', 'nvfix', 'amps', or 'json'.");
        }

        if(options.get("filter").hasValue())
        {
            filter = options.get("filter").getValue();
        }

        if(options.get("batchsize").hasValue())
        {
            batchSize = Integer.parseInt(options.get("batchsize").getValue());
        }

        if(options.get("bookmark").hasValue())
        {
            bookmark = options.get("bookmark").getValue();
        }
        if(options.get("orderby").hasValue())
        {
            orderBy = options.get("orderby").getValue();
        }
        if(options.get("opts").hasValue())
        {
            opts = options.get("opts").getValue();
        }
        if(options.get("topn").hasValue())
        {
            topN = Integer.parseInt(options.get("topn").getValue());
        }
        if(options.get("timeout").hasValue())
        {
            timeout = Integer.parseInt(options.get("timeout").getValue());
        }

        // Connect client to AMPS
        String userId = (userAndServer.length == 2) ? userAndServer[0] : System.getProperty("user.name");
        String clientName = "spark-sow-and-subscribe-" + userId + "-" + UUID.randomUUID().toString();
        Client client = new Client(clientName, Client.Version.AMPS_2);
        SOWAndSubscribeCommandHandler handler = new SOWAndSubscribeCommandHandler();

        client.setDisconnectHandler(handler);
        String uriString = null;
        try
        {
            uriString = "tcp://" + userId + "@" + server + "/" + type;
            client.connect(uriString);
            client.logon(5000,createAuthenticator(uriString));
        }
        catch (ConnectionException e)
        {
            System.err.println("Unable to connect to AMPS (" + e + ").");
            System.exit(1);
        }
        catch (Exception e)
        {
            System.err.println("Unable to connect to AMPS (" + e + ").");
            System.exit(1);
        }

        try
        {
            synchronized(handler)
            {
                Message message = client.allocateMessage();

                if(delta)
                {
                    message.setCommand(Message.Command.SOWAndDeltaSubscribe);
                }
                else
                {
                    message.setCommand(Message.Command.SOWAndSubscribe);
                }

                CommandId id = CommandId.nextIdentifier();
                message.setCommandId(id);
                message.setSubId(id);
                message.setQueryId(id);
                message.setAckType(Message.AckType.Processed);
                message.setTopic(topic);
                message.setBatchSize(batchSize);
                if (topN > 0) message.setTopN(topN);
                if (filter != null) message.setFilter(filter);
                if (orderBy != null) message.setOrderBy(orderBy);
                if (opts != null) message.setOptions(opts);
                if (bookmark != null) message.setBookmark(bookmark);

                // Execute the subscribe request, but only wait 5 seconds for it to be
                //   issued, otherwise we abandon it.
                client.send(handler,message,timeout);

                try
                {
                    handler.wait();  // This just hangs here until we're killed.
                }
                catch (InterruptedException e)
                {
                    // Nothing to do...
                }
            }
        }
        catch (Exception e)
        {
            // Failed to send
            System.err.println("Error sending subscribe request: " + e.toString());
        }

        client.close();
    }
}
