////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;

import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.fields.Field;
import com.crankuptheamps.client.fields.BookmarkField;


/**
 * RingBookmarkStore is an implementation of BookmarkStore that stores state on disk and in memory.
 * For each subscription, RingBookmarkStore keeps the most recent bookmark B for which all
 * messages prior to B have been discard()ed by the subscriber.  As messages are logged and
 * discarded, an in-memory array is used to track when a new bookmark can be logged.
 */
public class RingBookmarkStore implements BookmarkStore
{
    // Manages the portion of the store for a single subscription.
    // RingBookmarkStore uses a memory mapped file for persistent storage.
    // Each subscription has four elements: one subId of size BYTES_SUBID,
    // and three bookmarks of size BYTES_BOOKMARK.
    // On disk, each bookmark entry begins with one byte designating the next bookmark entry
    // to be written.  The value '*', aka the Cursor, in this position indicates this is the oldest bookmark
    // stored for the subscription, and is the next to be written.
    // Writing always follows a predictable sequence: the '*' for the next entry is written,
    // the bookmark id is written to the current entry, and then the cursor is erased for the current entry.
    // On recovery, the entry *before* the earliest one with the '*' is the most recently
    // written: in case of crash during write, there may be two entries with a '*', and the third one is the one to recover.

    // In memory, each Subscription has an array of entry's.  Each entry holds a bookmark and whether or not it has been discard()ed.
    // Whenever the 0th entry in the array is discarded, we know it's time to write a bookmark to disk.
    // We truncate the front of the array up through the last one that is inactive, and then write the next one to disk.

    static class Subscription implements com.crankuptheamps.client.Subscription
    {
        // The buffer holding the memory-mapped file of all subscriptions
        ByteBuffer _buffer;
        // The offset into _buffer where our subscription begins.
        int _offset;
        // The disk entry to be written next.
        short _currentDiskPosition = 0;

        BookmarkRingBuffer _ring = new BookmarkRingBuffer();

        // On-disk bytes for a bookmark entry.
        static final int BYTES_BOOKMARK = 32;
        // On-disk bytes for the subscription ID.
        static final int BYTES_SUBID = 32;
        // Number of bookmarks stored per subscription.  Must be at least 3 for recovery guarantee.
        static final short POSITIONS = 3;

        public Subscription()
        {
        }


        public void init(ByteBuffer buffer, int offset) throws AMPSException
        {
            _ring.noPersistedAcks();
            _buffer = buffer;
            _offset = offset;

            recover();
        }
        // A new bookmark has arrived for this subscription.  Just remember it:
        // nothing to write about until the message is discarded.
        public synchronized long log(BookmarkField bookmark)
        {
            return _ring.log(bookmark);
        }

        // This is never called but required by the interface
        public boolean isDiscarded(BookmarkField bookmark)
        {
            return false;
        }

        // User is finished with a bookmark.  Mark the entry as inactive,
        // and if it's the 0th entry, forward-truncate and log the most recent.
        public synchronized void discard(long bookmark)
        {
            // Calculate the position in _entries.
            if(_ring.discard(bookmark))
            {
                write(_ring.getLastDiscarded());
            }
        }

        public synchronized Field getMostRecent()
        {
            return _ring.getLastDiscarded();
        }

        public synchronized Field getMostRecentList()
        {
            return _ring.getLastDiscarded();
        }

        // Since we don't persist the discard()ed message that came after the most
        // recently logged bookmark, we always assume any message the user is testing for
        // should be delivered.
        public boolean isDiscarded(Field bookmark)
        {
            return false;
        }
        private void write(Field bookmark)
        {
            synchronized(this)
            {
                // We want to write to _currentDiskPosition.
                short nextDiskPosition = (short)((_currentDiskPosition + 1) % POSITIONS);

                // Mark the next position with the 'cursor'

                _buffer.put( _offset + BYTES_SUBID + (BYTES_BOOKMARK * nextDiskPosition), (byte)'*');

                // write the current position and validate it
                _buffer.position ( _offset + BYTES_SUBID + (BYTES_BOOKMARK * _currentDiskPosition) + 1 );
                for(int i = 0; i < bookmark.length; i++)
                {
                    _buffer.put(bookmark.byteAt(i));
                }
                for(int i = 0; i < BYTES_BOOKMARK - (bookmark.length+2); i++)
                {
                    _buffer.put((byte)0);
                }
                _buffer.put( _offset + BYTES_SUBID + (BYTES_BOOKMARK * _currentDiskPosition), (byte)'+');

                // advance _currentDiskPosition
                _currentDiskPosition = nextDiskPosition;
            }
        }

        private void recover()
        {
            // find the first cursor
            short foundCursor = 0;
            for(; foundCursor < POSITIONS; foundCursor++)
            {
                byte b = _buffer.get( _offset + BYTES_SUBID + (BYTES_BOOKMARK * foundCursor));
                if(b == (byte)'*') break;
            }
            if(foundCursor == 0)
            {
                byte b = _buffer.get(_offset + BYTES_SUBID + (BYTES_BOOKMARK * (POSITIONS - 1)));
                if(b==(byte)'*')
                {
                    foundCursor = POSITIONS - 1;
                }
            }

            if(foundCursor < POSITIONS)
            {
                // Found an existing "cursor": start the writing there.
                _currentDiskPosition = foundCursor;
                int mostRecentValid = _currentDiskPosition==0?POSITIONS-1:_currentDiskPosition-1;
                byte[] buf = new byte[BYTES_BOOKMARK-1];
                _buffer.position(_offset + BYTES_SUBID + (BYTES_BOOKMARK * mostRecentValid) + 1);
                _buffer.get(buf);

                int bookmarkLength = 0;
                for(; bookmarkLength< buf.length && buf[bookmarkLength] != 0; bookmarkLength++);

                try
                {
                    BookmarkField f = new BookmarkField();
                    f.set(buf, 0, bookmarkLength);
                    // discard and log to make this the "starting point" for the subscription.
                    _ring.discard(_ring.log(f));
                }
                catch (Exception e)
                {
                }
            }
            else
            {
                _currentDiskPosition = 0;
            }
        }

        public synchronized void persisted(BookmarkField bookmark)
        {
            // no-op
        }

        public synchronized void persisted(long bookmark)
        {
            // no-op
        }

        public void setLastPersisted(long bookmark)
        {
            // no-op
        }

        public void setLastPersisted(BookmarkField bookmark)
        {
            // no-op
        }

        public synchronized void noPersistedAcks()
        {
            // no-op
        }

        public synchronized void setPersistedAcks()
        {
            // no-op
        }

        public long getOldestBookmarkSeq()
        {
            return _ring.getStartIndex();
        }

        public void setResizeHandler(BookmarkStoreResizeHandler handler, BookmarkStore store)
        {
            _ring.setResizeHandler(handler, store);
        }
    }

    MappedByteBuffer _buffer;
    private static final int ENTRIES=16384;
    private static final int ENTRY_SIZE = Subscription.BYTES_SUBID + (Subscription.POSITIONS * Subscription.BYTES_BOOKMARK);
    private HashMap<Field, Subscription> _map;
    // Initialize to ENTRIES, so that there is no free space in the log until recover() has succeeded.
    private int _free = ENTRIES;
    RandomAccessFile _file;
    FileChannel _channel;
    String _path;
    BookmarkStoreResizeHandler _resizeHandler = null;
    private int _serverVersion = Client.MIN_MULTI_BOOKMARK_VERSION;

    Pool<Subscription> _pool;

    public RingBookmarkStore(String path) throws AMPSException
    {
        this(path, 1);
    }
    public RingBookmarkStore(String path, int targetNumberOfSubscriptions) throws AMPSException
    {
        try
        {
            _path = path;
            _file = new RandomAccessFile(_path, "rw");
            _pool = new Pool<Subscription>(Subscription.class, targetNumberOfSubscriptions);
        }
        catch (IOException ioex)
        {
            throw new AMPSException("I/O Error initializing file " + path, ioex);
        }
        init();
    }
    public long log(Message message) throws AMPSException
    {
        BookmarkField bookmark = (BookmarkField)message.getBookmarkRaw();
        Subscription sub = (RingBookmarkStore.Subscription)message.getSubscription();
        if (sub == null)
        {
            Field subId = message.getSubIdRaw();
            if (subId == null || subId.isNull())
                subId = message.getSubIdsRaw();
            sub = find(subId);
            message.setSubscription(sub);
        }
        long seqNo = sub.log(bookmark);
        message.setBookmarkSeqNo(seqNo);
        return seqNo;
    }

    public void discard(Field subId, long bookmarkSeqNo) throws AMPSException
    {
        find(subId).discard(bookmarkSeqNo);
    }

    public void discard(Message message) throws AMPSException
    {
        long bookmark = message.getBookmarkSeqNo();
        Subscription sub = (RingBookmarkStore.Subscription)message.getSubscription();
        if (sub == null)
        {
            Field subId = message.getSubIdRaw();
            if (subId == null || subId.isNull())
                subId = message.getSubIdsRaw();
            sub = find(subId);
            message.setSubscription(sub);
        }
        sub.discard(bookmark);
    }

    public Field getMostRecent(Field subId) throws AMPSException
    {
        return find(subId).getMostRecent();
    }

    public boolean isDiscarded(Message message) throws AMPSException
    {
        BookmarkField bookmark = (BookmarkField)message.getBookmarkRaw();
        Subscription sub = (RingBookmarkStore.Subscription)message.getSubscription();
        if (sub == null)
        {
            Field subId = message.getSubIdRaw();
            if (subId == null || subId.isNull())
                subId = message.getSubIdsRaw();
            sub = find(subId);
            message.setSubscription(sub);
        }
        return sub.isDiscarded(bookmark);
    }

    private void recover() throws AMPSException
    {
        int currentEntry = 0;
        for(; currentEntry < ENTRIES; currentEntry++)
        {
            // is it active?
            byte firstByte = _buffer.get(currentEntry * ENTRY_SIZE);
            if(firstByte != 0)
            {
                // read it out
                byte[] id = new byte[Subscription.BYTES_SUBID - 1];
                _buffer.position(currentEntry * ENTRY_SIZE);
                _buffer.get(id);
                int idLength = 0;
                for(; idLength < id.length && id[idLength] != 0; idLength++);

                Field f = new Field(id, 0, idLength);
                Subscription subscription = _pool.get();
                subscription.init(_buffer, currentEntry * ENTRY_SIZE);
                try
                {
                    _map.put(f, subscription);
                }
                catch (Exception e)
                {
                    throw new AMPSException("Bookmark store corrupted.", e);
                }
                subscription.recover();
            }
            else break;
        }
        if(currentEntry == ENTRIES)
        {
            // todo: resize here
            throw new AMPSException("Unable to allocate space in this bookmark store.");
        }
        _free = currentEntry;
    }

    private synchronized Subscription find(Field subId) throws AMPSException
    {
        if(_map.containsKey(subId))
        {
            return _map.get(subId);
        }

        if(_free >= ENTRIES)
        {
            throw new AMPSException("Unable to allocate space in this bookmark store.");
        }
        int pos = _free++;

        Subscription subscription = _pool.get();
        subscription.init(_buffer, pos * ENTRY_SIZE);
        subscription.setResizeHandler(_resizeHandler, this);
        _map.put(subId.copy(), subscription);
        _buffer.position(pos * ENTRY_SIZE);
        for(int i = 0; i < subId.length; i++)
        {
            _buffer.put(subId.buffer[subId.position+i]);
        }
        return subscription;
    }

    public void persisted(Field subId, BookmarkField bookmark) throws AMPSException
    {
        // no-op
    }

    public void persisted(Field subId, long bookmark) throws AMPSException
    {
        // no-op
    }

    public void noPersistedAcks(Field subId) throws AMPSException
    {
        // no-op
    }

    public long getOldestBookmarkSeq(Field subId) throws AMPSException
    {
        return find(subId).getOldestBookmarkSeq();
    }

    public void setResizeHandler(BookmarkStoreResizeHandler handler)
    {
        _resizeHandler = handler;
        Iterator it = _map.entrySet().iterator();
        while (it.hasNext())
        {
	    Map.Entry pairs = (Map.Entry)it.next();
	    ((Subscription)pairs.getValue()).setResizeHandler(handler, this);
        }
    }

    public synchronized void purge() throws AMPSException
    {
        // This is the quickest way I've found, so far, to zero out the file.
        int longs = _buffer.capacity()/8;
        _buffer.position(0);
        for(int i =0; i < longs; i++)
        {
            _buffer.putLong(0);
        }
        _buffer.position(0);

        _buffer.force();
        _map = new HashMap<Field,Subscription>();
        _free = 0;
        recover();
    }

    private void init() throws AMPSException
    {
        try
        {
            _channel = _file.getChannel();
            _buffer = _channel.map(FileChannel.MapMode.READ_WRITE, 0, ENTRIES*ENTRY_SIZE);
            _channel.close();
            _file.close();
        }
        catch (IOException ioex)
        {
            throw new AMPSException("error opening store.", ioex);
        }

        _map = new HashMap<Field,Subscription>();
        _free = 0;
        recover();
    }

    public void setServerVersion(int version)
    {
        _serverVersion = version;
    }
}
