package com.crankuptheamps.examples;

import java.lang.Thread;
import java.io.*;

import com.crankuptheamps.client.*;
import com.crankuptheamps.client.exception.*;

/**
 * SOWConsoleQuerier
 *
 * This sample retrieves messages from a state-of-the-world database. The
 * program flow is simple:
 *
 * * Connect to AMPS
 * * Logon
 * * Get the state-of-the-world for the "messages-sow" topic, filtered
 * messages with a message number less than 10.
 * * Output all messages received to the console
 *
 * This sample doesn't include error handling or connection
 * retry logic.
 *
 * (c) 2013-2015 60East Technologies, Inc. All rights reserved.
 * This file is a part of the AMPS Evaluation Kit.
 */
public class SOWConsoleQuerier {

  // The location of the AMPS server.
  private static final String uri_ = "tcp://127.0.0.1:9007/amps/json";

  /**
   * main method.
   *
   * @param args -- No command line options read.
   */

  public static void main(String[] args) {

    try (HAClient client = new HAClient("SOWConsoleQuerier");) {

      // Create a custom server chooser
      DefaultServerChooser sc = new DefaultServerChooser();
      sc.add(uri_);

      // Set the server chooser for the HAClient
      client.setServerChooser(sc);
      // connect to the AMPS server and logon
      client.connectAndLogon();
      System.out.println("SOW Querier Connected");

      // request messages from the messages-sow topic where
      // the messageNumber field is less than 10.

      try (MessageStream ms = client.execute(new Command(Message.Command.SOW)
          .setTopic("messages-sow").setFilter("/messageNumber < 10"));) {
        for (Message m : ms) {
          if (m.getCommand() == Message.Command.GroupBegin) {
            System.out.println("Receiving messages from SOW " +
                "(beginning of group).");
            continue;
          }
          if (m.getCommand() == Message.Command.GroupEnd) {
            System.out.println("Finished receiving messages from"
                + " SOW (end of group).");
            continue;
          }
          System.out.println("Message received: " + m.getData());
        }
      }
    } catch (AMPSException e) {
      System.err.println(e.getLocalizedMessage());
      e.printStackTrace(System.err);
    }

  }

}
