package com.crankuptheamps.examples;

import java.lang.Thread;

import com.crankuptheamps.client.*;
import com.crankuptheamps.client.exception.*;

/**
 * SowAndSubscribeWithOOF
 *
 * This sample retrieves messages from a state-of-the-world database. The
 * program flow is simple:
 *
 * * Connect to AMPS
 * * Logon
 * * Get the state-of-the-world for the "messages-sow" topic, filtered
 * to messages with a message number that is an even multiple of 10
 * and either no optionalField or an optionalField value other than
 * 'ignore_me'.
 * * Output all messages received to the console
 *
 * This sample doesn't include error handling or connection
 * retry logic.
 *
 * (c) 2013-2015 60East Technologies, Inc. All rights reserved.
 * This file is a part of the AMPS Evaluation Kit.
 */
public class SowAndSubscribeWithOOF  {

  // The location of the AMPS server.
  private static final String uri_ = "tcp://127.0.0.1:9007/amps/json";

  /**
   * main method.
   *
   * @param args -- No command line options read.
   */

  public static void main(String[] args) {

    try (HAClient client = new HAClient("SOWandSubscribeConsoleSubscribeWithOOF");) {

      // Create a custom server chooser
      DefaultServerChooser serverChooser = new DefaultServerChooser();
      serverChooser.add(uri_);
      // Set the server chooser for the HAClient
      client.setServerChooser(serverChooser);
      // connect to the AMPS server and logon
      client.connectAndLogon();
      System.out.println("SOW and Subscribe Console Subscribe with OOF Connected");

      // create the object to process the messages.
      SOWMessagePrinter smp = new SOWMessagePrinter();

      System.out.println("... entering subscription ...");

      // request messages from the messages-sow topic where
      // the message number is mod 10, and optionalField is
      // either missing or, if present, is not 'ignore_me'.

      // For sample purposes, we use the asynchronous form of execute.
      client.executeAsync(new Command(Message.Command.SOWAndSubscribe)
                              .setTopic("messages-sow")
                              .setFilter("/messageNumber % 10 = 0 AND " +
                                         "(/optionalField is NULL OR " +
                                         " /optionalField <> 'ignore_me')")
          .setOptions("oof"), smp);

      System.out.println("Requested SOW messages.");

      // the results of the sow query arrive asynchronously,
      // so the sample sleeps to let the messages arrive.

      // this program uses this construct for sample purposes.
      // generally speaking, the program would use the results
      // of the query as they arrive.

      while (true)
      {
        Thread.sleep(100);
      }

    }
    catch (AMPSException e) {
      System.err.println(e.getLocalizedMessage());
      e.printStackTrace(System.err);
    }
    catch (InterruptedException e) {
      System.err.println(e.getLocalizedMessage());
      e.printStackTrace(System.err);
    }
  }

}
