############################################################################
##
## Copyright (c) 2012-2013 60East Technologies Inc., All Rights Reserved.
##
## This computer software is owned by 60East Technologies Inc. and is
## protected by U.S. copyright laws and other laws and by international
## treaties.  This computer software is furnished by 60East Technologies
## Inc. pursuant to a written license agreement and may be used, copied,
## transmitted, and stored only in accordance with the terms of such
## license agreement and with the inclusion of the above copyright notice.
## This computer software or any other copies thereof may not be provided
## or otherwise made available to any other person.
##
## U.S. Government Restricted Rights.  This computer software: (a) was
## developed at private expense and is in all respects the proprietary
## information of 60East Technologies Inc.; (b) was not developed with
## government funds; (c) is a trade secret of 60East Technologies Inc.
## for all purposes of the Freedom of Information Act; and (d) is a
## commercial item and thus, pursuant to Section 12.212 of the Federal
## Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
## Government's use, duplication or disclosure of the computer software
## is subject to the restrictions set forth by 60East Technologies Inc..
##
############################################################################

__version__ = "3.1.0.7.218042.6dac798"
import sys

# Before doing anything, check that we're running at least version 2.5
if(''.join(map(str, sys.version_info[0:3])) < '250'):
    raise Exception('Error: AMPS requires Python 2.5 or greater.\n')

#TODO: filter these for unused cruft
import socket
import struct
import time
import os
import threading
import collections
import mmap
import binascii
import re
import base64
import uuid
import urlparse


def LOG(s, *args):
    print s % args


def STATS(s, *args):
    print s % args


def WARNING(s, *args):
    print s % args


def ERROR(s, *args):
    print s % args


def enum(*sequential, **named):
    enums = dict(zip(sequential, sequential), **named)
    return type('Enum', (), enums)


class AMPSLogHandler:
    LogLevel = enum('none', 'debug', 'info', 'warning', 'error', 'critical')

    def __init__(self, level=LogLevel.none):
        self._level = level

    def level(self):
        return self._level

    def log(self, level, message, exception=None):
        pass

StreamState = enum('start', 'in_sow', 'end')
###########################################################################
##
## Exception Classes
##
##    Exception
##       |
##    AMPSException
##       |
##       --CommandError
##       |      |
##       |      --CommandTypeError: An invalid command type was specified.
##       |      |
##       |      --CommandTimedOut: Timed out awaiting acknowledgement of
##       |      |                  command execution.
##       |      --BadFilter:       Invalid content filter used in a
##       |      |                  subscription or query.
##       |      --BadRegexTopic:   Invalid regex topic used in a
##       |      |                  subscription or query.
##       |      --ClientNameInUse: The client name designated is already in use.
##       |      |                  AMPS requires a unique client name at the
##       |      |                  time that a logon is created.
##       |      --SubscriptionAlreadyExists: A subscription with the identifier
##       |      |                  specified already exists.
##       |      --TimedOut:        The command timed out before it could be completed.
##       |      |
##       |      --UnknownError:    UnknownError occured (please report)
##       |
##       --ConnectionError
##       |      |
##       |      --ConnectionRefused: A connection could not be established.
##       |      |
##       |      --Disconnected:      An operation was attemped on a
##       |      |                    disconnected connection.
##       |      --AlreadyConnected:  Connect was called when already
##       |      |                    connected; call disconnect first.
##       |      --RetryOperation:    A disconnect event was handled during the operation
##       |      |                    and the transport reconnected successfully. OK to retry.
##       |      --InvalidUriFormat:  Invalid URI format specified.
##       |      |
##       |      --StreamError:       The message stream coming from AMPS
##       |      |                    wasn't formatted as expected.
##       |      --AuthenticationError:  An authentication error occured
##       |      |
##       |      --NotEntitledError : An entitlement error occured.
##       |      |
##       |      --TransportError
##       |      |        |
##       |      |        --TransportNotFound    Specified transport type couldn't be found.
##       |      |        |
##       |      |        --InvalidTransportOptions Invalid transport options were
##       |      |                                  used in the URI specification.
##       |      --MessageTypeError
##       |               |
##       |               --MessageTypeNotFound  Specified message type couldn't be found.
##       |               |
##       |               --InvalidMessageTypeOptions  Invalid message type options were
##       |                                            used in the URI specification.
##       --LocalStorageError
##       |      |
##       |      --CorruptedRecord: A corrupted record was found in a *local* store.
##
###########################################################################


class AMPSException(Exception):
    def __init__(self, *args):
        Exception.__init__(self, args)


class CommandError(AMPSException):
    def __init__(self, *args):
        AMPSException.__init__(self, args)


class CommandTypeError(CommandError):
    def __init__(self, commandType):
        self.commandType = commandType

    def __str__(self):
        return "Command type cannot be sent directly to AMPS (%s)" % self.commandType


class CommandTimedOut(CommandError):
    def __init__(self, commandId):
        self.commandId = commandId

    def __str__(self):
        return "command timed out (%s)" % self.commandId


class BadFilter(CommandError):
    def __init__(self, filter):
        self.filter = filter

    def __str__(self):
        return "bad content filter (%s)" % self.filter


class BadRegexTopic(CommandError):
    def __init__(self, topic):
        self.topic = topic

    def __str__(self):
        return "bad regex topic (%s)" % self.topic


class SubscriptionAlreadyExists(CommandError):
    def __init__(self, commandId):
        self.commandId = commandId

    def __str__(self):
        return "subscription already exists (%s)" % self.commandId


class TimedOut(CommandError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "operation timed out (%s)" % self.reason


class UnknownError(CommandError):
    def __init__(self, reason):
        #import pdb; pdb.set_trace()
        self.reason = reason

    def __str__(self):
        return "unknown error (%s)" % self.reason


class ConnectionError(AMPSException):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "connection error (%s)" % self.reason


class ConnectionRefused(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "connection refused (%s)" % self.reason


class AuthenticationError(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "Logon failed for user '%s'" % self.reason


class NotEntitledError(ConnectionError):
    def __init__(self, user, topic):
        self.user = user
        self.topic = topic

    def __str__(self):
        return "User \"%s\" not entitled to topic \"%s\"." % (self.user, self.topic)


class Disconnected(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "disconnected (%s)" % self.reason


class AlreadyConnected(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "already connected (%s)" % self.reason


class RetryOperation(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "retry operation (%s)" % self.reason


class InvalidUriFormat(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "invalid uri format (%s)" % self.reason


class TransportError(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "transport error (%s)" % self.reason


class TransportNotFound(TransportError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "transport not found (%s)" % self.reason


class InvalidTransportOptions(TransportError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "invalid transport options (%s)" % self.reason


class StreamError(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "stream error (%s)" % self.reason


class MessageTypeError(ConnectionError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "message type error (%s)" % self.reason


class MessageTypeNotFound(MessageTypeError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "message type not found (%s)" % self.reason


class InvalidMessageTypeOptions(MessageTypeError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "invalid message type options (%s)" % self.reason


class LocalStorageError(AMPSException):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "local storage error (%s)" % self.reason


class CorruptedRecord(LocalStorageError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "corrupted record error (%s)" % self.reason


class ClientNameInUse(CommandError):
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        return "client name in use error (%s)" % self.reason
class DefaultAuthenticator(object):
    def authenticate(self, username, currentPassword):
        return currentPassword

    def retry(self, username, password):
        return password

    def completed(self, username, password, reason):
        pass
############################################################################
###
### URI parser
###
##    Client connections follow the URI scheme:
##       transport://username:password@host:port/messageType?params
##
##    Example:
##       tcp://prod:token@localhost:9004/fix?fieldSeparator=1
##
############################################################################


class URI:
    def __init__(self, uri):
        self._uri = uri
        # This is a bit of cheat, since the "http" scheme covers our exact
        #   needs, then we'll switch the transport for http and then do the
        #   parse as normal.
        try:
            self._transport, r = uri.split(':', 1)
            o = urlparse.urlparse('http:' + r)
            self._messageType = o.path.strip('/')
            self._username = o.username
            self._password = o.password
            self._hostname = o.hostname
            self._port = int(o.port)
            self._args = {}
            if(len(o.query)):
                self._args = dict(
                    [kv.split('=') for kv in o.query.split('&')])
        except Exception, e:
            raise InvalidUriFormat(e)

    def uri(self):
        return self._uri

    def transport(self):
        return self._transport

    def messageType(self):
        return self._messageType

    def username(self):
        return self._username

    def password(self):
        return self._password

    def hostname(self):
        return self._hostname

    def port(self):
        return self._port

    def args(self):
        return self._args

    def asString(self):
        return "URI(transport=%s,messageType=%s,username=%s,password=%s,hostname=%s,port=%s,args=%r)" % \
            (self._transport, self._messageType, self._username,
             self._password, self._hostname, self._port, self._args)

#print URI("tcp://localhost:9004/fix?fieldSeparator=1").asString()
#print URI("tcp://localhost:9004/fix/?fieldSeparator=1").asString()
#print URI("tcp://localhost:9004/fix/?fieldSeparator=1&headerSeparator=2").asString()
#print URI("tcp://brand@localhost:9004/fix/?fieldSeparator=1&headerSeparator=2").asString()
#print URI("tcp://brand:password@localhost:9004/fix/?fieldSeparator=1&headerSeparator=2").asString()
###########################################################################
##
## Message classes
##
##    Command:         Class containing valid Message command types.
##    Ack:             Class containing valid Message ack types.
##    Status:          Class containing valid Message status values.
##    Reason:          Class containing valid Message reason values.
##    Message:         Base message class, can't be used for anything in
##                     Python.
##    InboundMessage:  Read-only message object containing a raw byte buffer.
##    OutboundMessage: User-friendly,Read-Write message object used for sending
##                     messages to AMPS.
##
###########################################################################


class Command:
    Unknown = 'unknown'
    Publish = 'publish'
    Subscribe = 'subscribe'
    Unsubscribe = 'unsubscribe'
    SOW = 'sow'
    Heartbeat = 'heartbeat'
    Logon = 'logon'
    StartTimer = 'start_timer'
    StopTimer = 'stop_timer'
    SOWAndSubscribe = 'sow_and_subscribe'
    DeltaPublish = 'delta_publish'
    DeltaSubscribe = 'delta_subscribe'
    SOWAndDeltaSubscribe = 'sow_and_delta_subscribe'
    SOWDelete = 'sow_delete'
    GroupBegin = 'group_begin'
    GroupEnd = 'group_end'
    OOF = 'oof'
    Ack = 'ack'


class AckType:
    NoAck = 0
    Received = 1
    Parsed = 2
    Persisted = 4
    Processed = 8
    Completed = 16
    Stats = 32

    # Translates a numeric id to string
    N2S_TRANSLATOR = {0: 'none',
                      1: 'received',
                      2: 'parsed',
                      4: 'persisted',
                      8: 'processed',
                      16: 'completed',
                      32: 'stats',
                      3: 'received,parsed',
                      5: 'received,persisted',
                      9: 'received,processed',
                      17: 'received,completed',
                      33: 'received,stats',
                      6: 'parsed,persisted',
                      10: 'parsed,processed',
                      18: 'parsed,completed',
                      34: 'parsed,stats',
                      12: 'persisted,processed',
                      20: 'persisted,completed',
                      36: 'persisted,stats',
                      24: 'processed,completed',
                      40: 'processed,stats',
                      48: 'completed,stats',
                      7: 'received,parsed,persisted',
                      11: 'received,parsed,processed',
                      19: 'received,parsed,completed',
                      35: 'received,parsed,stats',
                      13: 'received,persisted,processed',
                      21: 'received,persisted,completed',
                      37: 'received,persisted,stats',
                      25: 'received,processed,completed',
                      41: 'received,processed,stats',
                      49: 'received,completed,stats',
                      14: 'parsed,persisted,processed',
                      22: 'parsed,persisted,completed',
                      38: 'parsed,persisted,stats',
                      26: 'parsed,processed,completed',
                      42: 'parsed,processed,stats',
                      50: 'parsed,completed,stats',
                      28: 'persisted,processed,completed',
                      44: 'persisted,processed,stats',
                      52: 'persisted,completed,stats',
                      56: 'processed,completed,stats',
                      15: 'received,parsed,persisted,processed',
                      23: 'received,parsed,persisted,completed',
                      39: 'received,parsed,persisted,stats',
                      27: 'received,parsed,processed,completed',
                      43: 'received,parsed,processed,stats',
                      51: 'received,parsed,completed,stats',
                      29: 'received,persisted,processed,completed',
                      45: 'received,persisted,processed,stats',
                      53: 'received,persisted,completed,stats',
                      57: 'received,processed,completed,stats',
                      30: 'parsed,persisted,processed,completed',
                      46: 'parsed,persisted,processed,stats',
                      54: 'parsed,persisted,completed,stats',
                      58: 'parsed,processed,completed,stats',
                      60: 'persisted,processed,completed,stats',
                      31: 'received,parsed,persisted,processed,completed',
                      47: 'received,parsed,persisted,processed,stats',
                      55: 'received,parsed,persisted,completed,stats',
                      59: 'received,parsed,processed,completed,stats',
                      61: 'received,persisted,processed,completed,stats',
                      62: 'parsed,persisted,processed,completed,stats',
                      63: 'received,parsed,persisted,processed,completed,stats'}

    # Translates a string to a numeric id
    S2N_TRANSLATOR = dict(map(lambda t: (t[1], t[0]), N2S_TRANSLATOR.items()))

    @staticmethod
    def toString(command):
        return AckType.N2S_TRANSLATOR[command]

    @staticmethod
    def toAckType(name):
        return AckType.S2N_TRANSLATOR[name]


class Status:
    Success = 'success'
    Failure = 'failure'
    Retry = 'retry'


class Reason:
    Duplicate = 'duplicate'
    BadFilter = 'bad filter'
    BadRegexTopic = 'bad regex topic'
    SubscriptionAlreadyExists = 'subscription already exists'
    NameInUse = 'name in use'
    AuthFailure = 'auth failure'
    NotEntitled = 'not entitled'
    AuthDisabled = 'authentication disabled'


class Message(object):
    def __init__(self):
        self.reset()

    def reset(self):
        self._command = Command.Unknown
        self._commandId = None
        self._clientName = None
        self._userId = None
        self._timestamp = None
        self._topic = None
        self._filter = None
        self._ackType = None
        self._subId = None
        self._expiration = None
        self._subIds = None
        self._reason = None
        self._sendMatchingIds = None
        self._status = None
        self._queryId = None
        self._sendOOF = None
        self._batchSize = None
        self._topN = None
        self._sendEmpties = None
        self._sowDeleted = None
        self._topicMatches = None
        self._matches = None
        self._maxMessages = None
        self._sowKeys = None
        self._size = 0
        self._bookmark = None
        self._password = None
        self._options = None
        self._sequence = None
        self._sowKey = None
        self._groupSeqNo = None
        self._data = None

    def _invalidPropertyRead(self):
        raise NotImplementedError("Message is not readable")

    def _invalidPropertyWrite(self):
        raise NotImplementedError("Message is not writable")

    def getCommand(self):
        self._invalidPropertyRead()

    def setCommand(self, v):
        self._invalidPropertyWrite()

    def getCommandId(self):
        self._invalidPropertyRead()

    def setCommandId(self, v):
        self._invalidPropertyWrite()

    def getClientName(self):
        self._invalidPropertyRead()

    def setClientName(self, v):
        self._invalidPropertyWrite()

    def getUserId(self):
        self._invalidPropertyRead()

    def setUserId(self, v):
        self._invalidPropertyWrite()

    def getTimestamp(self):
        self._invalidPropertyRead()

    def setTimestamp(self, v):
        self._invalidPropertyWrite()

    def getTopic(self):
        self._invalidPropertyRead()

    def setTopic(self, v):
        self._invalidPropertyWrite()

    def getFilter(self):
        self._invalidPropertyRead()

    def setFilter(self, v):
        self._invalidPropertyWrite()

    def getAckType(self):
        self._invalidPropertyRead()

    def setAckType(self, v):
        self._invalidPropertyWrite()

    def getSubId(self):
        self._invalidPropertyRead()

    def setSubId(self, v):
        self._invalidPropertyWrite()

    def getExpiration(self):
        self._invalidPropertyRead()

    def setExpiration(self, v):
        self._invalidPropertyWrite()

    def getSendMatchingIds(self):
        self._invalidPropertyRead()

    def setSendMatchingIds(self, v):
        self._invalidPropertyWrite()

    def getSubIds(self):
        self._invalidPropertyRead()

    def setSubIds(self, v):
        self._invalidPropertyWrite()

    def getStatus(self):
        self._invalidPropertyRead()

    def setStatus(self, v):
        self._invalidPropertyWrite()

    def getQueryId(self):
        self._invalidPropertyRead()

    def setQueryId(self, v):
        self._invalidPropertyWrite()

    def getSendOOF(self):
        self._invalidPropertyRead()

    def setSendOOF(self, v):
        self._invalidPropertyWrite()

    def getBatchSize(self):
        self._invalidPropertyRead()

    def setBatchSize(self, v):
        self._invalidPropertyWrite()

    def getTopN(self):
        self._invalidPropertyRead()

    def setTopN(self, v):
        self._invalidPropertyWrite()

    def getSendEmpties(self):
        self._invalidPropertyRead()

    def setSendEmpties(self, v):
        self._invalidPropertyWrite()

    def getSowDeleted(self):
        self._invalidPropertyRead()

    def setSowDeleted(self, v):
        self._invalidPropertyWrite()

    def getTopicMatches(self):
        self._invalidPropertyRead()

    def setTopicMatches(self, v):
        self._invalidPropertyWrite()

    def getMatches(self):
        self._invalidPropertyRead()

    def setMatches(self, v):
        self._invalidPropertyWrite()

    def getMaxMessages(self):
        self._invalidPropertyRead()

    def setMaxMessages(self, v):
        self._invalidPropertyWrite()

    def getSowKey(self):
        self._invalidPropertyRead()

    def setSowKey(self, v):
        self._invalidPropertyWrite()

    def getSowKeys(self):
        self._invalidPropertyRead()

    def setSowKeys(self, v):
        self._invalidPropertyWrite()

    def getGroupSeqNo(self):
        self._invalidPropertyRead()

    def setGroupSeqNo(self, v):
        self._invalidPropertyWrite()

    def getSize(self):
        self._invalidPropertyRead()

    def setSize(self, v):
        self._invalidPropertyWrite()

    def getSequence(self):
        self._invalidPropertyRead()

    def setSequence(self, v):
        self._invalidPropertyWrite()

    def getPassword(self):
        self._invalidPropertyRead()

    def setPassword(self, v):
        self._invalidPropertyWrite()

    def getOptions(self):
        self._invalidPropertyRead()

    def setOptions(self, v):
        self._invalidPropertyWrite()

    def getData(self):
        self._invalidPropertyRead()

    def setData(self, v):
        self._invalidPropertyWrite()

    def __str__(self):
        return 'Message[%r]' % dict(filter(lambda t: t[1], self.__dict__.items()))


class OutboundMessage(Message):
    def __init__(self):
        Message.__init__(self)

    def getCommand(self):
        # Slower get, because we want the string here for fast serialization
        return self._command

    def setCommand(self, v):
        self._command = str(v)

    def getCommandId(self):
        return self._commandId

    def setCommandId(self, v):
        self._commandId = [None, str(v)][bool(v)]

    def getClientName(self):
        return self._clientName

    def setClientName(self, v):
        self._clientName = [None, str(v)][bool(v)]

    def getUserId(self):
        return self._userId

    def setUserId(self, v):
        self._userId = [None, str(v)][bool(v)]

    def getTimestamp(self):
        return self._timestamp

    def setTimestamp(self, v):
        self._timestamp = [None, str(v)][bool(v)]

    def getTopic(self):
        return self._topic

    def setTopic(self, v):
        self._topic = [None, str(v)][bool(v)]

    def getFilter(self):
        return self._filter

    def setFilter(self, v):
        self._filter = [None, str(v)][bool(v)]

    def getAckType(self):
        return self._ackType

    def setAckType(self, v):
        self._ackType = int(v)

    def getSubId(self):
        return self._subId

    def setSubId(self, v):
        self._subId = [None, str(v)][bool(v)]

    def getExpiration(self):
        return self._expiration

    def setExpiration(self, v):
        self._expiration = [None, str(v)][bool(v)]

    def getSendMatchingIds(self):
        return self._sendMatchingIds

    def setSendMatchingIds(self, v):
        if(v == 'true' or v == True):
            self._sendMatchingIds = 'true'
        else:
            self._sendMatchingIds = None

    def getSubIds(self):
        return self._subIds

    def setSubIds(self, v):
        self._subIds = [None, str(v)][bool(v)]

    def getReason(self):
        return self._reason

    def setReason(self, v):
        self._reason = [None, str(v)][bool(v)]

    def getStatus(self):
        return self._status

    def setStatus(self, v):
        self._status = [None, str(v)][bool(v)]

    def getQueryId(self):
        return self._queryId

    def setQueryId(self, v):
        self._queryId = [None, str(v)][bool(v)]

    def getSendOOF(self):
        return self._sendOOF

    def setSendOOF(self, v):
        if(v == 'true' or v == True):
            self._sendOOF = 'true'
        else:
            self._sendOOF = None

    def getBatchSize(self):
        return self._batchSize

    def setBatchSize(self, v):
        self._batchSize = [0, str(v)][bool(v)]

    def getTopN(self):
        return self._topN

    def setTopN(self, v):
        self._topN = [0, str(v)][bool(v)]

    def getSendEmpties(self):
        return self._sendEmpties

    def setSendEmpties(self, v):
        if(v == 'true' or v == True):
            self._sendOOF = 'true'
        else:
            self._sendOOF = None

    def getMaxMessages(self):
        return self._maxMessages

    def setMaxMessages(self, v):
        self._maxMessages = [0, str(v)][bool(v)]

    def getSowKey(self):
        return self._sowKey

    def setSowKey(self, v):
        self._sowKey = [None, str(v)][bool(v)]

    def getSowKeys(self):
        return self._sowKeys

    def setSowKeys(self, v):
        self._sowKeys = [None, str(v)][bool(v)]

    def getGroupSeqNo(self):
        return self._groupSeqNo

    def setGroupSeqNo(self, v):
        self._groupSeqNo = [None, str(v)][bool(v)]

    def getSize(self):
        return int(self._size)

    def setSize(self, v):
        self._size = [0, str(v)][bool(v)]

    def getBookmark(self):
        return self._bookmark

    def setBookmark(self, v):
        self._bookmark = [None, str(v)][v is not None]

    def getPassword(self):
        return self._password

    def setPassword(self, v):
        self._password = [None, str(v)][bool(v)]

    def getOptions(self):
        return self._password

    def getSequence(self):
        return int(self._sequence)

    def setSequence(self, v):
        self._sequence = [None, str(v)][bool(v)]

    def getData(self):
        return self._data

    def setData(self, v):
        self._data = v

    def __str__(self):
        return 'OutboundMessage[%r]' % dict(filter(lambda t: t[1], self.__dict__.items()))


class InboundMessage(Message):
    def __init__(self):
        Message.__init__(self)

    def reset(self):
        Message.reset(self)
        self._rawBuffer = None
        self._rawBufferOffset = None
        self._rawBufferLength = None

    def getRawBuffer(self):
        return self._rawBuffer

    def setRawBuffer(self, buffer):
        self._rawBuffer = buffer

    def getRawBufferOffset(self):
        return self._rawBufferOffset

    def setRawBufferOffset(self, offset):
        self._rawBufferOffset = offset

    def getRawBufferLength(self):
        return self._rawBufferOffset

    def setRawBufferLength(self, length):
        self._rawBufferOffset = length

    def getCommand(self):
        return self._command

    def _setCommand(self, v):
        self._command = str(v)

    def getCommandId(self):
        return self._commandId

    def _setCommandId(self, v):
        self._commandId = v

    def getClientName(self):
        return self._clientName

    def _setClientName(self, v):
        self._clientName = v

    def getUserId(self):
        return self._userId

    def _setUserId(self, v):
        self._userId = v

    def getTimestamp(self):
        return self._timestamp

    def _setTimestamp(self, v):
        self._timestamp = v

    def getTopic(self):
        return self._topic

    def _setTopic(self, v):
        self._topic = v

    def getFilter(self):
        return self._filter

    def _setFilter(self, v):
        self._filter = v

    def getAckType(self):
        return self._ackType

    def _setAckType(self, v):
        self._ackType = AckType.toAckType(v)

    def getSubId(self):
        return self._subId

    def _setSubId(self, v):
        self._subId = v

    def getExpiration(self):
        return self._expiration

    def _setExpiration(self, v):
        self._expiration = v

    def getSendMatchingIds(self):
        return self._sendMatchingIds

    def _setSendMatchingIds(self, v):
        self._sendMatchingIds = v

    def getSubIds(self):
        return self._subIds

    def _setSubIds(self, v):
        self._subIds = v

    def getReason(self):
        return self._reason

    def _setReason(self, v):
        self._reason = v

    def getStatus(self):
        return self._status

    def _setStatus(self, v):
        self._status = v

    def getQueryId(self):
        return self._queryId

    def _setQueryId(self, v):
        self._queryId = v

    def getSendOOF(self):
        return self._sendOOF

    def _setSendOOF(self, v):
        self._sendOOF = v

    def getBatchSize(self):
        return self._batchSize

    def _setBatchSize(self, v):
        self._batchSize = int(v)

    def getTopN(self):
        return self._topN

    def _setTopN(self, v):
        self._topN = int(v)

    def getSendEmpties(self):
        return self._sendEmpties

    def _setSendEmpties(self, v):
        self._sendEmpties = int(v)

    def getSowDeleted(self):
        return self._sowDeleted

    def _setSowDeleted(self, v):
        self._sowDeleted = int(v)

    def getTopicMatches(self):
        return self._topicMatches

    def _setTopicMatches(self, v):
        self._topicMatches = int(v)

    def getMatches(self):
        return self._matches

    def _setMatches(self, v):
        self._matches = int(v)

    def getMaxMessages(self):
        return self._maxMessages

    def _setMaxMessages(self, v):
        self._maxMessages = int(v)

    def getSowKey(self):
        return self._sowKey

    def _setSowKey(self, v):
        self._sowKey = v

    def getSowKeys(self):
        return self._sowKeys

    def _setSowKeys(self, v):
        self._sowKeys = v

    def getGroupSeqNo(self):
        return self._groupSeqNo

    def _setGroupSeqNo(self, v):
        self._groupSeqNo = v

    def getSize(self):
        return self._size

    def _setSize(self, v):
        self._size = int(v)

    def getBookmark(self):
        return self._bookmark

    def _setBookmark(self, v):
        self._bookmark = v

    def getPassword(self):
        return self._password

    def _setPassword(self, v):
        self._password = v

    def getOptions(self):
        return self._options

    def _setOptions(self, v):
        self._options = v

    def getSequence(self):
        return self._sequence

    def _setSequence(self, v):
        self._sequence = int(v)

    def getData(self):
        return self._data

    def _setData(self, v):
        self._data = v

    def __str__(self):
        return 'InboundMessage[%r]' % dict(filter(lambda t: t[1], self.__dict__.items()))
class MessageTypeFactory:
    Registry = {}

    @staticmethod
    def register(name, messageTypeProducer):
        # Only allow re-registry to the same producer
        assert(name not in MessageTypeFactory.Registry or
               (MessageTypeFactory.Registry[name] == messageTypeProducer))
        MessageTypeFactory.Registry[name] = messageTypeProducer
        return True

    @staticmethod
    def unregister(name):
        if(name in MessageTypeFactory.Registry):
            del MessageTypeFactory.Registry[name]
        return True

    @staticmethod
    def createMessageType(name, options):
        if(name in MessageTypeFactory.Registry):
            return MessageTypeFactory.Registry[name](options)
        raise MessageTypeNotFound(name)
class FIXHeaderFields:
    COMMAND = '20000='
    COMMAND_ID = '20001='
    CLIENT_NAME = '20002='
    USER_ID = '20003='
    TIMESTAMP = '20004='
    TOPIC = '20005='
    FILTER = '20006='
    ACK_TYPE = '20008='
    SUB_ID = '20009='
    EXPIRATION = '20012='
    SEND_MATCHING_IDS = '20013='
    STATUS = '20018='
    QUERY_ID = '20019='
    SEND_OOF = '20020='
    BATCH_SIZE = '20023='
    TOPN = '20025='
    SEND_EMPTIES = '20029='
    MAX_MESSAGES = '20031='
    SOW_KEYS = '20032='
    CLIENT_SEQUENCE = '20036='
    BOOKMARK = '20037='
    PASSWORD = '20038='
    OPTIONS = '20039='
    SOW_DELETED = '20054='
    TOPIC_MATCHES = '20056='
    MATCHES = '20057='
    MESSAGE_SIZE = '20058='
    SOW_KEY = '20059='
    GROUP_SEQ_NO = '20060='
    SUB_IDS = '20061='
    REASON = '20062='


class FIXFSPlusHeaderFields:
    def __init__(self, fieldSeparator, headerSeparator):
        self.FSHS = '%s%s' % (fieldSeparator, headerSeparator)
        self.FSCOMMAND = '%s%s' % (
            fieldSeparator, FIXHeaderFields.COMMAND)
        self.FSCOMMAND_ID = '%s%s' % (
            fieldSeparator, FIXHeaderFields.COMMAND_ID)
        self.FSCLIENT_NAME = '%s%s' % (
            fieldSeparator, FIXHeaderFields.CLIENT_NAME)
        self.FSUSER_ID = '%s%s' % (
            fieldSeparator, FIXHeaderFields.USER_ID)
        self.FSTIMESTAMP = '%s%s' % (
            fieldSeparator, FIXHeaderFields.TIMESTAMP)
        self.FSTOPIC = '%s%s' % (
            fieldSeparator, FIXHeaderFields.TOPIC)
        self.FSFILTER = '%s%s' % (
            fieldSeparator, FIXHeaderFields.FILTER)
        self.FSACK_TYPE = '%s%s' % (
            fieldSeparator, FIXHeaderFields.ACK_TYPE)
        self.FSSUB_ID = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SUB_ID)
        self.FSEXPIRATION = '%s%s' % (
            fieldSeparator, FIXHeaderFields.EXPIRATION)
        self.FSSEND_MATCHING_IDS = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SEND_MATCHING_IDS)
        self.FSSTATUS = '%s%s' % (
            fieldSeparator, FIXHeaderFields.STATUS)
        self.FSQUERY_ID = '%s%s' % (
            fieldSeparator, FIXHeaderFields.QUERY_ID)
        self.FSSEND_OOF = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SEND_OOF)
        self.FSBATCH_SIZE = '%s%s' % (
            fieldSeparator, FIXHeaderFields.BATCH_SIZE)
        self.FSTOPN = '%s%s' % (
            fieldSeparator, FIXHeaderFields.TOPN)
        self.FSSEND_EMPTIES = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SEND_EMPTIES)
        self.FSMAX_MESSAGES = '%s%s' % (
            fieldSeparator, FIXHeaderFields.MAX_MESSAGES)
        self.FSSOW_KEYS = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SOW_KEYS)
        self.FSCLIENT_SEQUENCE = '%s%s' % (
            fieldSeparator, FIXHeaderFields.CLIENT_SEQUENCE)
        self.FSBOOKMARK = '%s%s' % (
            fieldSeparator, FIXHeaderFields.BOOKMARK)
        self.FSPASSWORD = '%s%s' % (
            fieldSeparator, FIXHeaderFields.PASSWORD)
        self.FSOPTIONS = '%s%s' % (
            fieldSeparator, FIXHeaderFields.OPTIONS)
        self.FSSOW_DELETED = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SOW_DELETED)
        self.FSTOPIC_MATCHES = '%s%s' % (
            fieldSeparator, FIXHeaderFields.TOPIC_MATCHES)
        self.FSMATCHES = '%s%s' % (
            fieldSeparator, FIXHeaderFields.MATCHES)
        self.FSMESSAGE_SIZE = '%s%s' % (
            fieldSeparator, FIXHeaderFields.MESSAGE_SIZE)
        self.FSSOW_KEY = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SOW_KEY)
        self.FSGROUP_SEQ_NO = '%s%s' % (
            fieldSeparator, FIXHeaderFields.GROUP_SEQ_NO)
        self.FSSUB_IDS = '%s%s' % (
            fieldSeparator, FIXHeaderFields.SUB_IDS)
        self.FSREASON = '%s%s' % (
            fieldSeparator, FIXHeaderFields.REASON)


class FIXMessageStream:
    HeaderFieldSetterMap = {
        FIXHeaderFields.COMMAND: InboundMessage._setCommand,
        FIXHeaderFields.COMMAND_ID: InboundMessage._setCommandId,
        FIXHeaderFields.CLIENT_NAME: InboundMessage._setClientName,
        FIXHeaderFields.USER_ID: InboundMessage._setUserId,
        FIXHeaderFields.TIMESTAMP: InboundMessage._setTimestamp,
        FIXHeaderFields.TOPIC: InboundMessage._setTopic,
        FIXHeaderFields.FILTER: InboundMessage._setFilter,
        FIXHeaderFields.ACK_TYPE: InboundMessage._setAckType,
        FIXHeaderFields.SUB_ID: InboundMessage._setSubId,
        FIXHeaderFields.EXPIRATION: InboundMessage._setExpiration,
        FIXHeaderFields.SEND_MATCHING_IDS: InboundMessage._setSendMatchingIds,
        FIXHeaderFields.STATUS: InboundMessage._setStatus,
        FIXHeaderFields.QUERY_ID: InboundMessage._setQueryId,
        FIXHeaderFields.SEND_OOF: InboundMessage._setSendOOF,
        FIXHeaderFields.BATCH_SIZE: InboundMessage._setBatchSize,
        FIXHeaderFields.TOPN: InboundMessage._setTopN,
        FIXHeaderFields.SEND_EMPTIES: InboundMessage._setSendEmpties,
        FIXHeaderFields.SOW_DELETED: InboundMessage._setSowDeleted,
        FIXHeaderFields.TOPIC_MATCHES: InboundMessage._setTopicMatches,
        FIXHeaderFields.MATCHES: InboundMessage._setMatches,
        FIXHeaderFields.MAX_MESSAGES: InboundMessage._setMaxMessages,
        FIXHeaderFields.SOW_KEYS: InboundMessage._setSowKeys,
        FIXHeaderFields.CLIENT_SEQUENCE: InboundMessage._setSequence,
        FIXHeaderFields.BOOKMARK: InboundMessage._setBookmark,
        FIXHeaderFields.PASSWORD: InboundMessage._setPassword,
        FIXHeaderFields.OPTIONS: InboundMessage._setOptions,
        FIXHeaderFields.MESSAGE_SIZE: InboundMessage._setSize,
        FIXHeaderFields.SOW_KEY: InboundMessage._setSowKey,
        FIXHeaderFields.GROUP_SEQ_NO: InboundMessage._setGroupSeqNo,
        FIXHeaderFields.SUB_IDS: InboundMessage._setSubIds,
        FIXHeaderFields.REASON: InboundMessage._setReason}

    def __init__(self, messageType):
        self._messageType = messageType
        self._remainingBytes = 0
        self._message = InboundMessage()
        self._buffer = None
        self.HeaderFieldRegEx = re.compile(
            r'(200\d+=)(.*?)%s' % messageType.fieldSeparator)
        self._fixHeaderFields = FIXFSPlusHeaderFields(
            messageType.fieldSeparator, messageType.headerSeparator)

    def process(self, buf, pos, size, listener):
        self._buffer = buf
        self._pos = pos
        self._remainingBytes = size
        self._state = StreamState.start
        self._recordsRemaining = 0
        self._message.reset()
        self._message.setRawBuffer(self._buffer)
        messagesRemaining = self.read(self._message)
        listener(self._message)
        while(messagesRemaining):
            messagesRemaining = self.read(self._message)
            listener(self._message)

    def peekByte(self):
        return self._buffer[self._pos]

    def advance(self, n):
        self._pos += n
        self._remainingBytes -= n

    def remainingBytes(self):
        return self._remainingBytes

    def getByte(self):
        b = self._buffer[self._pos]
        self.advance(1)
        return b

    def getBytes(self, n):
        s = self._buffer[self._pos:self._pos + n]
        self.advance(n)
        return s

    def _extractFields(self, message, hsloc):
        hfm = FIXMessageStream.HeaderFieldSetterMap
        try:
            for tag, value in self.HeaderFieldRegEx.findall(self._buffer, self._pos, hsloc):
                sfield = hfm.get(tag)
                sfield(message, value)
        except Exception:
            import traceback
            traceback.print_exc()
            raise StreamError(
                "stream corruption: unknown header tag found ('%s')" % tag)
        self.advance(hsloc - self._pos + 1)  # +1 for header sep

    def read(self, message):
        if(self._state == StreamState.start):
            hs = self._messageType.headerSeparator
            # Find the end of the header
            end = self._buffer.find(hs, self._pos)
            self._extractFields(message, end)

            if(message.getCommand() == Command.SOW):
                end = self._buffer.find(hs, self._pos)
                self._extractFields(message, end)
                messageLength = message.getSize()
                message.setRawBufferOffset(self._pos)
                message.setRawBufferLength(messageLength)
                message._setData(
                    self._buffer[self._pos:self._pos + messageLength])
                self._recordsRemaining = message.getBatchSize() - 1
                if(self._recordsRemaining > 0):
                    # Advance to next sow record
                    self.advance(messageLength + 1)  # +1 for msg sep
                    self._state = StreamState.in_sow
                else:
                    # Advance to end
                    self.advance(self.remainingBytes())
            elif(message.getCommand() == Command.Publish):
                end = self._buffer.find(hs, self._pos)
                self._extractFields(message, end)
                messageLength = message.getSize()
                message.setRawBufferOffset(self._pos)
                message.setRawBufferLength(messageLength)
                message._setData(
                    self._buffer[self._pos:self._pos + messageLength])
                self.advance(self.remainingBytes())
            else:
                # Not a 'sow' or publish message, just a standard message
                messageLength = self.remainingBytes()  # No length here
                message.setRawBufferOffset(self._pos)
                message.setRawBufferLength(messageLength)
                message._setSize(messageLength)
                message._setData(
                    self._buffer[self._pos:self._pos + messageLength])
                self.advance(self.remainingBytes())
        elif(self._state == StreamState.in_sow):
            end = self._buffer.find(
                self._messageType.headerSeparator, self._pos)
            self._extractFields(message, end)
            messageLength = message.getSize()
            message.setRawBufferOffset(self._pos)
            message.setRawBufferLength(messageLength)
            message._setData(self._buffer[self._pos:self._pos + messageLength])
            self._recordsRemaining -= 1
            if(self._recordsRemaining > 0):
                # Advance to next sow record
                self.advance(messageLength + 1)  # +1 for msg sep
                self._state = StreamState.in_sow
            else:
                # Advance to end
                self.advance(self.remainingBytes())
        return self._recordsRemaining

    def serialize(self, message):
        # construct header, the method is mind-numbing, but
        #  using a function to simplify the code eats cycles
        header = []
        ma = header.append  # LOAD_ATTR avoidance (check by disassembly)
        fsphs = self._fixHeaderFields
        # We're leveraging the fact that command has to be present
        #   for the message to be correct. The command is laid down
        #   as normal, and then the special string values that contain
        #   field separators built in are used.
        ma(FIXHeaderFields.COMMAND)
        ma(message._command)
        if(message._commandId):
            ma(fsphs.FSCOMMAND_ID)
            ma(message._commandId)
        if(message._clientName):
            ma(fsphs.FSCLIENT_NAME)
            ma(message._clientName)
        if(message._userId):
            ma(fsphs.FSUSER_ID)
            ma(message._userId)
        if(message._timestamp):
            ma(fsphs.FSTIMESTAMP)
            ma(message._timestamp)
        if(message._topic):
            ma(fsphs.FSTOPIC)
            ma(message._topic)
        if(message._filter):
            ma(fsphs.FSFILTER)
            ma(message._filter)
        if(message._ackType):
            ma(fsphs.FSACK_TYPE)
            ma(AckType.toString(message._ackType))
        if(message._subId):
            ma(fsphs.FSSUB_ID)
            ma(message._subId)
        if(message._expiration):
            ma(fsphs.FSEXPIRATION)
            ma(message._expiration)
        if(message._sendMatchingIds):
            ma(fsphs.FSSEND_MATCHING_IDS)
            ma(message._sendMatchingIds)
        if(message._queryId):
            ma(fsphs.FSQUERY_ID)
            ma(message._queryId)
        if(message._sendOOF):
            ma(fsphs.FSSEND_OOF)
            ma(message._sendOOF)
        if(message._batchSize):
            ma(fsphs.FSBATCH_SIZE)
            ma(message._batchSize)
        if(message._topN):
            ma(fsphs.FSTOPN)
            ma(message._topN)
        if(message._sendEmpties):
            ma(fsphs.FSSEND_EMPTIES)
            ma(message._sendEmpties)
        if(message._maxMessages):
            ma(fsphs.FSMAX_MESSAGES)
            ma(message._maxMessages)
        if(message._sowKeys):
            ma(fsphs.FSSOW_KEYS)
            ma(message._sowKeys)
        if(message._sequence):
            ma(fsphs.FSCLIENT_SEQUENCE)
            ma(message._sequence)
        if(message._bookmark):
            ma(fsphs.FSBOOKMARK)
            ma(message._bookmark)
        if(message._password):
            ma(fsphs.FSPASSWORD)
            ma(message._password)
        if(message._options):
            ma(fsphs.FSOPTIONS)
            ma(message._options)
        if(message._sowKey):
            ma(fsphs.FSSOW_KEY)
            ma(message._sowKey)
        ma(fsphs.FSHS)
        # add body data
        if(message._data):
            ma(message._data)
        return ''.join(header)


class FIXMessageType:
    MessageTypeFactory.register(
        "fix", lambda args: FIXMessageType.createMessageType(args))

    def __init__(self, fieldSeparator=chr(1), headerSeparator=chr(2), messageSeparator=chr(3)):
        self.fieldSeparator = fieldSeparator
        self.headerSeparator = headerSeparator
        self.messageSeparator = messageSeparator

    @staticmethod
    def createMessageType(args):
        try:
            # defaults
            fs, hs, ms = chr(1), chr(2), chr(3)
            if("fieldSeparator" in args):
                fs = chr(int(args["fieldSeparator"]))
            if("headerSeparator" in args):
                hs = chr(int(args["headerSeparator"]))
            if("messageSeparator" in args):
                ms = chr(int(args["messageSeparator"]))
            return FIXMessageType(fs, hs, ms)
        except Exception, e:
            raise InvalidMessageTypeOptions(e)

    def stream(self):
        return FIXMessageStream(self)

    def allocateMessage(self):
        return OutboundMessage()
#############################################################################
##
## NVFIX Message Type
##
##   This message type is identical to the FIX message type on the client
##     client side (other than alphanumeric tags in the message body).
##
#############################################################################


class NVFIXMessageStream(FIXMessageStream):
    def __init__(self, messageType):
        FIXMessageStream.__init__(self, messageType)


class NVFIXMessageType(FIXMessageType):
    MessageTypeFactory.register(
        "nvfix", lambda args: NVFIXMessageType.createMessageType(args))

    def __init__(self, fieldSeparator=chr(1), headerSeparator=chr(2), messageSeparator=chr(3)):
        self.fieldSeparator = fieldSeparator
        self.headerSeparator = headerSeparator
        self.messageSeparator = messageSeparator

    @staticmethod
    def createMessageType(args):
        try:
            # defaults
            fs, hs, ms = chr(1), chr(2), chr(3)
            if("fieldSeparator" in args):
                fs = chr(int(args["fieldSeparator"]))
            if("headerSeparator" in args):
                hs = chr(int(args["headerSeparator"]))
            if("messageSeparator" in args):
                ms = chr(int(args["messageSeparator"]))
            return NVFIXMessageType(fs, hs, ms)
        except Exception, e:
            raise InvalidMessageTypeOptions(e)

    def stream(self):
        return NVFIXMessageStream(self)

    def allocateMessage(self):
        return OutboundMessage()
class XMLHeaderFields:
    EXPIRATION = 'Expn'
    EXPIRATION_A = '<Expn>'
    EXPIRATION_Z = '</Expn>'
    MAX_MESSAGES = 'MxMsgs'
    MAX_MESSAGES_A = '<MxMsgs>'
    MAX_MESSAGES_Z = '</MxMsgs>'
    USER_ID = 'UsrId'
    USER_ID_A = '<UsrId>'
    USER_ID_Z = '</UsrId>'
    SEND_EMPTIES = 'SndEmpty'
    SEND_EMPTIES_A = '<SndEmpty>'
    SEND_EMPTIES_Z = '</SndEmpty>'
    SEND_OOF = 'SndOOF'
    SEND_OOF_A = '<SndOOF>'
    SEND_OOF_Z = '</SndOOF>'
    TOPN = 'TopN'
    TOPN_A = '<TopN>'
    TOPN_Z = '</TopN>'
    PASSWORD = 'PW'
    PASSWORD_A = '<PW>'
    PASSWORD_Z = '</PW>'
    OPTIONS = 'Options'
    OPTIONS_A = '<Options>'
    OPTIONS_Z = '</Options>'
    COMMAND = 'Cmd'
    COMMAND_A = '<Cmd>'
    COMMAND_Z = '</Cmd>'
    COMMAND_ID = 'CmdId'
    COMMAND_ID_A = '<CmdId>'
    COMMAND_ID_Z = '</CmdId>'
    CLIENT_NAME = 'ClntName'
    CLIENT_NAME_A = '<ClntName>'
    CLIENT_NAME_Z = '</ClntName>'
    TIMESTAMP = 'TxmTm'
    TIMESTAMP_A = '<TxmTm>'
    TIMESTAMP_Z = '</TxmTm>'
    TOPIC = 'Tpc'
    TOPIC_A = '<Tpc>'
    TOPIC_Z = '</Tpc>'
    FILTER = 'Fltr'
    FILTER_A = '<Fltr>'
    FILTER_Z = '</Fltr>'
    ACK_TYPE = 'AckTyp'
    ACK_TYPE_A = '<AckTyp>'
    ACK_TYPE_Z = '</AckTyp>'
    SUB_ID = 'SubId'
    SUB_ID_A = '<SubId>'
    SUB_ID_Z = '</SubId>'
    SUB_IDS = 'SubIds'
    SUB_IDS_A = '<SubIds>'
    SUB_IDS_Z = '</SubIds>'
    SEND_MATCHING_IDS = 'SndSubIds'
    SEND_MATCHING_IDS_A = '<SndSubIds>'
    SEND_MATCHING_IDS_Z = '</SndSubIds>'
    STATUS = 'Status'
    STATUS_A = '<Status>'
    STATUS_Z = '</Status>'
    QUERY_ID = 'QId'
    QUERY_ID_A = '<QId>'
    QUERY_ID_Z = '</QId>'
    BATCH_SIZE = 'BtchSz'
    BATCH_SIZE_A = '<BtchSz>'
    BATCH_SIZE_Z = '</BtchSz>'
    CLIENT_SEQUENCE = 'Seq'
    CLIENT_SEQUENCE_A = '<Seq>'
    CLIENT_SEQUENCE_Z = '</Seq>'
    BOOKMARK = 'BkMrk'
    BOOKMARK_A = '<BkMrk>'
    BOOKMARK_Z = '</BkMrk>'
    MESSAGE_SIZE = 'MsgLn'
    MESSAGE_SIZE_A = '<MsgLn>'
    MESSAGE_SIZE_Z = '</MsgLn>'
    SOW_KEY = 'SowKey'
    SOW_KEY_A = '<SowKey>'
    SOW_KEY_Z = '</SowKey>'
    SOW_KEYS = 'SowKeys'
    SOW_KEYS_A = '<SowKeys>'
    SOW_KEYS_Z = '</SowKeys>'
    GROUP_SEQ_NO = 'GrpSqNum'
    GROUP_SEQ_NO_A = '<GrpSqNum>'
    GROUP_SEQ_NO_Z = '</GrpSqNum>'
    REASON = 'Reason'
    REASON_A = '<Reason>'
    REASON_Z = '</Reason>'
    SOW_DELETED = 'RecordsDeleted'
    SOW_DELETED_A = '<RecordsDeleted>'
    SOW_DELETED_Z = '</RecordsDeleted>'
    TOPIC_MATCHES = 'TopicMatches'
    TOPIC_MATCHES_A = '<TopicMatches>'
    TOPIC_MATCHES_Z = '</TopicMatches>'
    MATCHES = 'Matches'
    MATCHES_A = '<Matches>'
    MATCHES_Z = '</Matches>'


class XMLMessageStream:
    HeaderFieldSetterMap = {
        XMLHeaderFields.COMMAND: InboundMessage._setCommand,
        XMLHeaderFields.COMMAND_ID: InboundMessage._setCommandId,
        XMLHeaderFields.CLIENT_NAME: InboundMessage._setClientName,
        XMLHeaderFields.USER_ID: InboundMessage._setUserId,
        XMLHeaderFields.TIMESTAMP: InboundMessage._setTimestamp,
        XMLHeaderFields.TOPIC: InboundMessage._setTopic,
        XMLHeaderFields.FILTER: InboundMessage._setFilter,
        XMLHeaderFields.ACK_TYPE: InboundMessage._setAckType,
        XMLHeaderFields.SUB_ID: InboundMessage._setSubId,
        XMLHeaderFields.EXPIRATION: InboundMessage._setExpiration,
        XMLHeaderFields.SEND_MATCHING_IDS: InboundMessage._setSendMatchingIds,
        XMLHeaderFields.STATUS: InboundMessage._setStatus,
        XMLHeaderFields.QUERY_ID: InboundMessage._setQueryId,
        XMLHeaderFields.SEND_OOF: InboundMessage._setSendOOF,
        XMLHeaderFields.BATCH_SIZE: InboundMessage._setBatchSize,
        XMLHeaderFields.TOPN: InboundMessage._setTopN,
        XMLHeaderFields.SEND_EMPTIES: InboundMessage._setSendEmpties,
        XMLHeaderFields.MAX_MESSAGES: InboundMessage._setMaxMessages,
        XMLHeaderFields.CLIENT_SEQUENCE: InboundMessage._setSequence,
        XMLHeaderFields.BOOKMARK: InboundMessage._setBookmark,
        XMLHeaderFields.PASSWORD: InboundMessage._setPassword,
        XMLHeaderFields.OPTIONS: InboundMessage._setOptions,
        XMLHeaderFields.MESSAGE_SIZE: InboundMessage._setSize,
        XMLHeaderFields.SOW_KEY: InboundMessage._setSowKey,
        XMLHeaderFields.GROUP_SEQ_NO: InboundMessage._setGroupSeqNo,
        XMLHeaderFields.SUB_IDS: InboundMessage._setSubIds,
        XMLHeaderFields.SOW_DELETED: InboundMessage._setSowDeleted,
        XMLHeaderFields.TOPIC_MATCHES: InboundMessage._setTopicMatches,
        XMLHeaderFields.MATCHES: InboundMessage._setMatches,
        XMLHeaderFields.REASON: InboundMessage._setReason}

    def __init__(self, messageType):
        self._messageType = messageType
        self._remainingBytes = 0
        self._message = InboundMessage()
        self._buffer = None
        self.HeaderFieldRegEx = re.compile(r'<(\S+?)>(.*?)</\1>')
        self._xmlHeaderFields = XMLHeaderFields()
        self._recordsRemaining = 0

    def process(self, buf, pos, size, listener):
        self._buffer = buf
        self._pos = pos
        self._remainingBytes = size
        self._state = StreamState.start
        self._recordsRemaining = 0
        self._message.reset()
        self._message.setRawBuffer(self._buffer)
        messagesRemaining = self.read(self._message)
        listener(self._message)
        while(messagesRemaining):
            messagesRemaining = self.read(self._message)
            listener(self._message)

    def peekByte(self):
        return self._buffer[self._pos]

    def advance(self, n):
        self._pos += n
        self._remainingBytes -= n

    def advancePast(self, s):
        loc = self._buffer.find(s, self._pos)
        if(loc < 0):
            raise StreamError("stream corruption: message remainder did not contain '%s'" % s)
        # We found it, let's advance past it
        toAdvance = loc + len(s) - self._pos
        self.advance(toAdvance)

    def remainingBytes(self):
        return self._remainingBytes

    def getByte(self):
        b = self._buffer[self._pos]
        self.advance(1)
        return b

    def getBytes(self, n):
        s = self._buffer[self._pos:self._pos + n]
        self.advance(n)
        return s

    def _extractFields(self, message, hsloc):
        hfm = XMLMessageStream.HeaderFieldSetterMap
        try:
            for tag, value in self.HeaderFieldRegEx.findall(self._buffer, self._pos, hsloc):
                sfield = hfm.get(tag)
                sfield(message, value)
        except:
            raise StreamError(
                "stream corruption: unknown header tag found ('%s')" % tag)
        self.advance(hsloc - self._pos + 1)  # +1 for header sep

    def read(self, message):
        if(self._state == StreamState.start):
            # Skip to where the header fields start
            self.advancePast(':Header>')
            # Find the end of the header
            end = self._buffer.rfind('</', self._pos, self._buffer.find(
                ':Header>', self._pos, self.remainingBytes() + self._pos))
            self._extractFields(message, end)

            if(message.getCommand() == Command.SOW):
                self.advancePast(' key="')
                key = self._buffer[self._pos:self._buffer.find('"', self._pos)]
                self.advancePast(' len="')
                messageLength = int(
                    self._buffer[self._pos:self._buffer.find('"', self._pos)])
                self.advancePast('>')
                message._setSowKey(key)
                message._setSize(messageLength)
                message.setRawBufferOffset(self._pos)
                message.setRawBufferLength(messageLength)
                message._setData(
                    self._buffer[self._pos:self._pos + messageLength])
                self._recordsRemaining = int(message.getBatchSize()) - 1
                if(self._recordsRemaining > 0):
                    # Advance to next sow record
                    self.advance(messageLength + 6)  # len('</Msg>')
                    self._state = StreamState.in_sow
                else:
                    # Advance to end
                    self.advance(self.remainingBytes())
            else:
                # Not a 'sow' message, just a standard message
                messageLength = message.getSize()
                if(messageLength):
                    self.advancePast(':Body>')
                    message.setRawBufferOffset(self._pos)
                    message.setRawBufferLength(messageLength)
                    message._setData(
                        self._buffer[self._pos:self._pos + messageLength])
                    self._state = StreamState.end
                self.advance(self.remainingBytes())
        elif(self._state == StreamState.in_sow):
            self.advancePast(' key="')
            key = self._buffer[self._pos:self._buffer.find('"', self._pos)]
            self.advancePast(' len="')
            messageLength = int(
                self._buffer[self._pos:self._buffer.find('"', self._pos)])
            self.advancePast('>')
            message._setSowKey(key)
            message._setSize(messageLength)
            message.setRawBufferOffset(self._pos)
            message.setRawBufferLength(messageLength)
            message._setData(self._buffer[self._pos:self._pos + messageLength])
            self._recordsRemaining -= 1
            if(self._recordsRemaining > 0):
                # Advance to next sow record
                self.advance(messageLength + 6)  # len('</Msg>')
                self._state = StreamState.in_sow
            else:
                # Advance to end
                self.advance(self.remainingBytes())
        return self._recordsRemaining

    def serialize(self, message):
        # construct header, the method is mind-numbing, but
        #  using a function to simplify the code eats cycles
        header = ['<?xml version="1.0"?><SOAP-ENV:Envelope><SOAP-ENV:Header>']
        ma = header.append  # LOAD_ATTR avoidance (check by disassembly)
        fsphs = self._xmlHeaderFields
        ma(fsphs.COMMAND_A)
        ma(message._command)
        ma(fsphs.COMMAND_Z)
        if(message._commandId):
            ma(fsphs.COMMAND_ID_A)
            ma(message._commandId)
            ma(fsphs.COMMAND_ID_Z)
        if(message._clientName):
            ma(fsphs.CLIENT_NAME_A)
            ma(message._clientName)
            ma(fsphs.CLIENT_NAME_Z)
        if(message._userId):
            ma(fsphs.USER_ID_A)
            ma(message._userId)
            ma(fsphs.USER_ID_Z)
        if(message._timestamp):
            ma(fsphs.TIMESTAMP_A)
            ma(message._timestamp)
            ma(fsphs.TIMESTAMP_Z)
        if(message._topic):
            ma(fsphs.TOPIC_A)
            ma(message._topic)
            ma(fsphs.TOPIC_Z)
        if(message._filter):
            ma(fsphs.FILTER_A)
            ma(message._filter)
            ma(fsphs.FILTER_Z)
        if(message._ackType):
            ma(fsphs.ACK_TYPE_A)
            ma(AckType.toString(message._ackType))
            ma(fsphs.ACK_TYPE_Z)
        if(message._subId):
            ma(fsphs.SUB_ID_A)
            ma(message._subId)
            ma(fsphs.SUB_ID_Z)
        if(message._expiration):
            ma(fsphs.EXPIRATION_A)
            ma(message._expiration)
            ma(fsphs.EXPIRATION_Z)
        if(message._sendMatchingIds):
            ma(fsphs.SEND_MATCHING_IDS_A)
            ma(message._sendMatchingIds)
            ma(fsphs.SEND_MATCHING_IDS_Z)
        if(message._queryId):
            ma(fsphs.QUERY_ID_A)
            ma(message._queryId)
            ma(fsphs.QUERY_ID_Z)
        if(message._sendOOF):
            ma(fsphs.SEND_OOF_A)
            ma(message._sendOOF)
            ma(fsphs.SEND_OOF_Z)
        if(message._batchSize):
            ma(fsphs.BATCH_SIZE_A)
            ma(message._batchSize)
            ma(fsphs.BATCH_SIZE_Z)
        if(message._topN):
            ma(fsphs.TOPN_A)
            ma(message._topN)
            ma(fsphs.TOPN_Z)
        if(message._sendEmpties):
            ma(fsphs.SEND_EMPTIES_A)
            ma(message._sendEmpties)
            ma(fsphs.SEND_EMPTIES_Z)
        if(message._maxMessages):
            ma(fsphs.MAX_MESSAGES_A)
            ma(message._maxMessages)
            ma(fsphs.MAX_MESSAGES_Z)
        if(message._sequence):
            ma(fsphs.CLIENT_SEQUENCE_A)
            ma(message._sequence)
            ma(fsphs.CLIENT_SEQUENCE_Z)
        if(message._bookmark):
            ma(fsphs.BOOKMARK_A)
            ma(message._bookmark)
            ma(fsphs.BOOKMARK_Z)
        if(message._password):
            ma(fsphs.PASSWORD_A)
            ma(message._password)
            ma(fsphs.PASSWORD_Z)
        if(message._options):
            ma(fsphs.OPTIONS_A)
            ma(message._options)
            ma(fsphs.OPTIONS_Z)
        if(message._sowKey):
            ma(fsphs.SOW_KEY_A)
            ma(message._sowKey)
            ma(fsphs.SOW_KEY_Z)
        ma('</SOAP-ENV:Header><SOAP-ENV:Body>')
        # add body data
        if(message._data):
            ma(message._data)
        ma('</SOAP-ENV:Body></SOAP-ENV:Envelope>')
        return ''.join(header)


class XMLMessageType:
    MessageTypeFactory.register(
        "xml", lambda args: XMLMessageType.createMessageType(args))

    def __init__(self):
        pass

    @staticmethod
    def createMessageType(args):
        try:
            return XMLMessageType()
        except Exception, e:
            raise InvalidMessageTypeOptions(e)

    def stream(self):
        return XMLMessageStream(self)

    def allocateMessage(self):
        return OutboundMessage()
class AMPSHeaderFields:
    COMMAND = '00='
    COMMAND_ID = '01='
    CLIENT_NAME = '02='
    USER_ID = '03='
    TIMESTAMP = '04='
    TOPIC = '05='
    FILTER = '06='
    ACK_TYPE = '08='
    SUB_ID = '09='
    EXPIRATION = '10='
    SEND_MATCHING_IDS = '13='
    STATUS = '18='
    QUERY_ID = '19='
    SEND_OOF = '20='
    BATCH_SIZE = '23='
    TOPN = '25='
    SEND_EMPTIES = '29='
    MAX_MESSAGES = '31='
    SOW_KEYS = '32='
    CLIENT_SEQUENCE = '36='
    BOOKMARK = '37='
    PASSWORD = '38='
    OPTIONS = '39='
    SOW_DELETED = '54='
    TOPIC_MATCHES = '56='
    MATCHES = '57='
    MESSAGE_SIZE = '58='
    SOW_KEY = '59='
    GROUP_SEQ_NO = '60='
    SUB_IDS = '61='
    REASON = '62='


class AMPSFSPlusHeaderFields:
    def __init__(self, fieldSeparator, headerSeparator):
        self.FSHS = '%s%s' % (fieldSeparator, headerSeparator)
        self.FSCOMMAND = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.COMMAND)
        self.FSCOMMAND_ID = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.COMMAND_ID)
        self.FSCLIENT_NAME = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.CLIENT_NAME)
        self.FSUSER_ID = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.USER_ID)
        self.FSTIMESTAMP = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.TIMESTAMP)
        self.FSTOPIC = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.TOPIC)
        self.FSFILTER = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.FILTER)
        self.FSACK_TYPE = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.ACK_TYPE)
        self.FSSUB_ID = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SUB_ID)
        self.FSEXPIRATION = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.EXPIRATION)
        self.FSSEND_MATCHING_IDS = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SEND_MATCHING_IDS)
        self.FSSTATUS = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.STATUS)
        self.FSQUERY_ID = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.QUERY_ID)
        self.FSSEND_OOF = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SEND_OOF)
        self.FSBATCH_SIZE = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.BATCH_SIZE)
        self.FSTOPN = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.TOPN)
        self.FSSEND_EMPTIES = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SEND_EMPTIES)
        self.FSMAX_MESSAGES = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.MAX_MESSAGES)
        self.FSSOW_KEYS = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SOW_KEYS)
        self.FSCLIENT_SEQUENCE = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.CLIENT_SEQUENCE)
        self.FSBOOKMARK = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.BOOKMARK)
        self.FSPASSWORD = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.PASSWORD)
        self.FSOPTIONS = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.OPTIONS)
        self.FSSOW_DELETED = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SOW_DELETED)
        self.FSTOPIC_MATCHES = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.TOPIC_MATCHES)
        self.FSMATCHES = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.MATCHES)
        self.FSMESSAGE_SIZE = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.MESSAGE_SIZE)
        self.FSSOW_KEY = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SOW_KEY)
        self.FSGROUP_SEQ_NO = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.GROUP_SEQ_NO)
        self.FSSUB_IDS = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.SUB_IDS)
        self.FSREASON = '%s%s' % (
            fieldSeparator, AMPSHeaderFields.REASON)


class AMPSMessageStream:
    HeaderFieldSetterMap = {
        AMPSHeaderFields.COMMAND: InboundMessage._setCommand,
        AMPSHeaderFields.COMMAND_ID: InboundMessage._setCommandId,
        AMPSHeaderFields.CLIENT_NAME: InboundMessage._setClientName,
        AMPSHeaderFields.USER_ID: InboundMessage._setUserId,
        AMPSHeaderFields.TIMESTAMP: InboundMessage._setTimestamp,
        AMPSHeaderFields.TOPIC: InboundMessage._setTopic,
        AMPSHeaderFields.FILTER: InboundMessage._setFilter,
        AMPSHeaderFields.ACK_TYPE: InboundMessage._setAckType,
        AMPSHeaderFields.SUB_ID: InboundMessage._setSubId,
        AMPSHeaderFields.EXPIRATION: InboundMessage._setExpiration,
        AMPSHeaderFields.SEND_MATCHING_IDS: InboundMessage._setSendMatchingIds,
        AMPSHeaderFields.STATUS: InboundMessage._setStatus,
        AMPSHeaderFields.QUERY_ID: InboundMessage._setQueryId,
        AMPSHeaderFields.SEND_OOF: InboundMessage._setSendOOF,
        AMPSHeaderFields.BATCH_SIZE: InboundMessage._setBatchSize,
        AMPSHeaderFields.TOPN: InboundMessage._setTopN,
        AMPSHeaderFields.SEND_EMPTIES: InboundMessage._setSendEmpties,
        AMPSHeaderFields.SOW_DELETED: InboundMessage._setSowDeleted,
        AMPSHeaderFields.TOPIC_MATCHES: InboundMessage._setTopicMatches,
        AMPSHeaderFields.MATCHES: InboundMessage._setMatches,
        AMPSHeaderFields.MAX_MESSAGES: InboundMessage._setMaxMessages,
        AMPSHeaderFields.SOW_KEYS: InboundMessage._setSowKeys,
        AMPSHeaderFields.CLIENT_SEQUENCE: InboundMessage._setSequence,
        AMPSHeaderFields.BOOKMARK: InboundMessage._setBookmark,
        AMPSHeaderFields.PASSWORD: InboundMessage._setPassword,
        AMPSHeaderFields.OPTIONS: InboundMessage._setOptions,
        AMPSHeaderFields.MESSAGE_SIZE: InboundMessage._setSize,
        AMPSHeaderFields.SOW_KEY: InboundMessage._setSowKey,
        AMPSHeaderFields.GROUP_SEQ_NO: InboundMessage._setGroupSeqNo,
        AMPSHeaderFields.SUB_IDS: InboundMessage._setSubIds,
        AMPSHeaderFields.REASON: InboundMessage._setReason}

    def __init__(self, messageType):
        self._messageType = messageType
        self._remainingBytes = 0
        self._message = InboundMessage()
        self._buffer = None
        self.HeaderFieldRegEx = re.compile(
            r'(\d+=)(.*?)%s' % messageType.fieldSeparator)
        self._fixHeaderFields = AMPSFSPlusHeaderFields(
            messageType.fieldSeparator, messageType.headerSeparator)

    def process(self, buf, pos, size, listener):
        self._buffer = buf
        self._pos = pos
        self._remainingBytes = size
        self._state = StreamState.start
        self._recordsRemaining = 0
        self._message.reset()
        self._message.setRawBuffer(self._buffer)
        messagesRemaining = self.read(self._message)
        listener(self._message)
        while(messagesRemaining):
            messagesRemaining = self.read(self._message)
            listener(self._message)

    def peekByte(self):
        return self._buffer[self._pos]

    def advance(self, n):
        self._pos += n
        self._remainingBytes -= n

    def remainingBytes(self):
        return self._remainingBytes

    def getByte(self):
        b = self._buffer[self._pos]
        self.advance(1)
        return b

    def getBytes(self, n):
        s = self._buffer[self._pos:self._pos + n]
        self.advance(n)
        return s

    def _extractFields(self, message, hsloc):
        hfm = AMPSMessageStream.HeaderFieldSetterMap
        try:
            for tag, value in self.HeaderFieldRegEx.findall(self._buffer, self._pos, hsloc):
                sfield = hfm.get(tag)
                sfield(message, value)
        except Exception:
            import traceback
            traceback.print_exc()
            raise StreamError(
                "stream corruption: unknown header tag found ('%s')" % tag)
        self.advance(hsloc - self._pos + 1)  # +1 for header sep

    def read(self, message):
        if(self._state == StreamState.start):
            hs = self._messageType.headerSeparator
            # Find the end of the header
            end = self._buffer.find(hs, self._pos)
            self._extractFields(message, end)

            if(message.getCommand() == Command.SOW):
                end = self._buffer.find(hs, self._pos)
                self._extractFields(message, end)
                messageLength = message.getSize()
                message.setRawBufferOffset(self._pos)
                message.setRawBufferLength(messageLength)
                message._setData(
                    self._buffer[self._pos:self._pos + messageLength])
                self._recordsRemaining = message.getBatchSize() - 1
                if(self._recordsRemaining > 0):
                    # Advance to next sow record
                    self.advance(messageLength + 1)  # +1 for msg sep
                    self._state = StreamState.in_sow
                else:
                    # Advance to end
                    self.advance(self.remainingBytes())
            elif(message.getCommand() == Command.Publish):
                end = self._buffer.find(hs, self._pos)
                self._extractFields(message, end)
                messageLength = message.getSize()
                message.setRawBufferOffset(self._pos)
                message.setRawBufferLength(messageLength)
                message._setData(
                    self._buffer[self._pos:self._pos + messageLength])
                self.advance(self.remainingBytes())
            else:
                # Not a 'sow' or publish message, just a standard message
                messageLength = self.remainingBytes()  # No length here
                message.setRawBufferOffset(self._pos)
                message.setRawBufferLength(messageLength)
                message._setSize(messageLength)
                message._setData(
                    self._buffer[self._pos:self._pos + messageLength])
                self.advance(self.remainingBytes())
        elif(self._state == StreamState.in_sow):
            end = self._buffer.find(
                self._messageType.headerSeparator, self._pos)
            self._extractFields(message, end)
            messageLength = message.getSize()
            message.setRawBufferOffset(self._pos)
            message.setRawBufferLength(messageLength)
            message._setData(self._buffer[self._pos:self._pos + messageLength])
            self._recordsRemaining -= 1
            if(self._recordsRemaining > 0):
                # Advance to next sow record
                self.advance(messageLength + 1)  # +1 for msg sep
                self._state = StreamState.in_sow
            else:
                # Advance to end
                self.advance(self.remainingBytes())
        return self._recordsRemaining

    def serialize(self, message):
        # construct header, the method is mind-numbing, but
        #  using a function to simplify the code eats cycles
        header = []
        ma = header.append  # LOAD_ATTR avoidance (check by disassembly)
        fsphs = self._fixHeaderFields
        # We're leveraging the fact that command has to be present
        #   for the message to be correct. The command is laid down
        #   as normal, and then the special string values that contain
        #   field separators built in are used.
        ma(AMPSHeaderFields.COMMAND)
        ma(message._command)
        if(message._commandId):
            ma(fsphs.FSCOMMAND_ID)
            ma(message._commandId)
        if(message._clientName):
            ma(fsphs.FSCLIENT_NAME)
            ma(message._clientName)
        if(message._userId):
            ma(fsphs.FSUSER_ID)
            ma(message._userId)
        if(message._timestamp):
            ma(fsphs.FSTIMESTAMP)
            ma(message._timestamp)
        if(message._topic):
            ma(fsphs.FSTOPIC)
            ma(message._topic)
        if(message._filter):
            ma(fsphs.FSFILTER)
            ma(message._filter)
        if(message._ackType):
            ma(fsphs.FSACK_TYPE)
            ma(AckType.toString(message._ackType))
        if(message._subId):
            ma(fsphs.FSSUB_ID)
            ma(message._subId)
        if(message._expiration):
            ma(fsphs.FSEXPIRATION)
            ma(message._expiration)
        if(message._sendMatchingIds):
            ma(fsphs.FSSEND_MATCHING_IDS)
            ma(message._sendMatchingIds)
        if(message._queryId):
            ma(fsphs.FSQUERY_ID)
            ma(message._queryId)
        if(message._sendOOF):
            ma(fsphs.FSSEND_OOF)
            ma(message._sendOOF)
        if(message._batchSize):
            ma(fsphs.FSBATCH_SIZE)
            ma(message._batchSize)
        if(message._topN):
            ma(fsphs.FSTOPN)
            ma(message._topN)
        if(message._sendEmpties):
            ma(fsphs.FSSEND_EMPTIES)
            ma(message._sendEmpties)
        if(message._maxMessages):
            ma(fsphs.FSMAX_MESSAGES)
            ma(message._maxMessages)
        if(message._sowKeys):
            ma(fsphs.FSSOW_KEYS)
            ma(message._sowKeys)
        if(message._sequence):
            ma(fsphs.FSCLIENT_SEQUENCE)
            ma(message._sequence)
        if(message._bookmark):
            ma(fsphs.FSBOOKMARK)
            ma(message._bookmark)
        if(message._password):
            ma(fsphs.FSPASSWORD)
            ma(message._password)
        if(message._options):
            ma(fsphs.FSOPTIONS)
            ma(message._options)
        if(message._sowKey):
            ma(fsphs.FSSOW_KEY)
            ma(message._sowKey)
        ma(fsphs.FSHS)
        # add body data
        if(message._data):
            ma(message._data)
        return ''.join(header)


class AMPSMessageType:
    MessageTypeFactory.register(
        "amps", lambda args: AMPSMessageType.createMessageType(args))

    def __init__(self, fieldSeparator=chr(1), headerSeparator=chr(2), messageSeparator=chr(3)):
        self.fieldSeparator = fieldSeparator
        self.headerSeparator = headerSeparator
        self.messageSeparator = messageSeparator

    @staticmethod
    def createMessageType(args):
        try:
            # defaults
            fs, hs, ms = chr(1), chr(2), chr(3)
            if("fieldSeparator" in args):
                fs = chr(int(args["fieldSeparator"]))
            if("headerSeparator" in args):
                hs = chr(int(args["headerSeparator"]))
            if("messageSeparator" in args):
                ms = chr(int(args["messageSeparator"]))
            return AMPSMessageType(fs, hs, ms)
        except Exception, e:
            raise InvalidMessageTypeOptions(e)

    def stream(self):
        return AMPSMessageStream(self)

    def allocateMessage(self):
        return OutboundMessage()
class TransportFactory:
    Registry = {}

    @staticmethod
    def register(name, transportProducer):
        # Only allow re-registry to the same producer
        assert(name not in TransportFactory.Registry or
               (TransportFactory.Registry[name] == transportProducer))
        TransportFactory.Registry[name] = transportProducer
        return True

    @staticmethod
    def unregister(name):
        if(name in TransportFactory.Registry):
            del TransportFactory.Registry[name]
        return True

    @staticmethod
    def createTransport(name, messageType, options):
        if(name in TransportFactory.Registry):
            return TransportFactory.Registry[name](messageType, options)
        raise TransportNotFound(name)
class TCPReaderThread(threading.Thread):
    READ_SIZE = 16 * 1024

    def __init__(self, transport):
        threading.Thread.__init__(self)
        self._transport = transport
        self.daemon = True
        self.start()
        self._stopped = False

    def stop(self):
        self._stopped = True

    def run(self):
        while(True):
            try:
                if(self._stopped):
                    return
                self._transport._handleReadEvent()
            except:
                pass


class TCPTransportImpl:
    def __init__(self, messageStream, onMessage, onDisconnect):
        self._addr = None
        self._socket = None
        self._lock = threading.Lock()
        self._rfifo = ''
        self._connectCounter = 0
        self._disconnectCounter = 0
        self._messageStream = messageStream
        self._onMessage = onMessage
        self._onDisconnect = onDisconnect
        self._REventLoop = None

    def rob(self, transport):
        # We're in a reconnect state where
        #   other threads may be waiting on our lock for the
        #   disconnect handler to finish.
        # Therefore, we steal all members except the lock,
        #   we keep our potentially locked lock, so that we
        #   can just release it once reconnected.
        with transport._lock:
            self._addr = transport._addr
            self._socket = transport._socket
            self._rfifo = transport._rfifo
            self._connectCounter = transport._connectCounter
            self._disconnectCounter = transport._disconnectCounter
            self._messageStream = transport._messageStream
            self._REventLoop = transport._REventLoop

    def connect(self, host, port, args):
        with self._lock:
            if(self._addr):
                raise AlreadyConnected(
                    'already connected to %s:%s' % self._addr)
            try:
                addr = (host, port)
                self._socket = socket.socket(
                    socket.AF_INET, socket.SOCK_STREAM)
                if args is not None:
                    for val in args:
                        try:
                            if val == "tcp_rcvbuf":
                                self._socket.setsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF, int(args[val]))
                            elif val == "tcp_keepalive":
                                self._socket.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, int(args[val]))
                            elif val == "tcp_sndbuf":
                                self._socket.setsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF, int(args[val]))
                            elif val == "tcp_nodelay":
                                self._socket.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, int(args[val]))
                            elif val == "tcp_linger":
                                self._socket.setsockopt(socket.SOL_SOCKET, socket.SO_LINGER, struct.pack("ii", 1, int(args[val])))
                        except:
                            raise InvalidUriFormat(
                                "invalid value for option %s" % val)

                self._socket.connect(addr)
                self._socket.settimeout(.1)
                self._rfifo = ''
                self._REventLoop = TCPReaderThread(self)
                self._addr = addr
                self._connectCounter += 1
            except socket.error, e:
                self._disconnect()
                raise ConnectionRefused(e[1])

    def _isConnected(self):
        return self._connectCounter > self._disconnectCounter

    def _disconnect(self):
        try:
            if(self._addr):
                self._disconnectCounter = self._connectCounter
                #import traceback; traceback.print_stack()
                self._REventLoop.stop()
                # We need to ensure the reader thread is done before
                # we return from here.
                self._socket.shutdown(socket.SHUT_RDWR)
                self._socket.close()
                self._lock.release()
                try:
                    self._REventLoop.join()
                except:
                    pass
                self._lock.acquire()
                self._REventLoop = None

        except Exception:
            pass  # nothing gets out
        self._rfifo = ''
        self._addr = None

    def disconnect(self):
        with self._lock:
            self._disconnect()

    def send(self, buf):
        # We've previously drained everything from the
        #  queue, so let's be optimistic and try to
        #  write this too.
        try:
            sent = 0
            tosend = len(buf)
            while(sent < tosend):
                try:
                    sent += self._socket.send(buf[sent:])
                except socket.timeout:
                    continue
        except AttributeError:
            raise Disconnected("disconnected")
        except socket.error:
            # Bad file descriptor,broken pipe,econnreset
            raise Disconnected("disconnected")

    def writeQueueSize(self):
        return 0

    def readQueueSize(self):
        return 0

    def flush(self, timeout):
        with self._lock:
            return

    def _handleReadEvent(self):
        # read more data into the rfifo
        try:
            data = self._socket.recv(16 * 1024)
            if(len(data) == 0):
                self.handleCloseEvent()
            self._rfifo += data
        except socket.timeout:
            pass
        except socket.error:
            self.handleCloseEvent()

        # now that we have a big buffer, let's read until we can't read
        # a full message
        length = len(self._rfifo)
        pos = 0
        while(True):
            if(length - pos < 4):
                # not enough room for size header, bail
                break
            msize = socket.ntohl(struct.unpack_from("I", self._rfifo, pos)[0])
            if(length - pos - 4 < msize):
                # we don't have the entire message, bail
                break
            pos += 4
            # TODO: What happens when the handler throws?
            #(answer: we should catch, and provide on error interface.)
            try:
                self._messageStream.process(
                    self._rfifo, pos, msize, self._onMessage)
            except Exception, e:
                ERROR("Caught exception from onMessage handler: %s", e)
            pos += msize
        # We're done, let's save remaining bytes for later
        self._rfifo = self._rfifo[pos:]

    def handleCloseEvent(self):
        with self._lock:
            # This can only be executed once per successful connection attempt.
            #   This assumes the lock is held.
            if(self._connectCounter > self._disconnectCounter):
                try:
                    self._disconnect()
                    self._onDisconnect()
                except Exception, e:
                    ERROR("Caught exception from onDisconnect handler: %s", e)
                    self._disconnect()
            if(self._isConnected()):
                raise RetryOperation("reconnected")
            else:
                raise Disconnected("reconnect failed")


class TCPTransport:
    TransportFactory.register("tcp", lambda messageType, options: TCPTransport.createTransport(messageType, options))

    @staticmethod
    def __default_on_disconnect_handler(c):
        raise Disconnected('no handler')

    def __init__(self, messageType):
        self._messageType = messageType
        self._messageStream = messageType.stream()
        self._impl = TCPTransportImpl(
            self._messageStream, self._onMessage, self._onDisconnect)
        self._onDisconnectHandler = self.__default_on_disconnect_handler
        self._onMessageHandler = lambda m: None

    @staticmethod
    def createTransport(messageType, options):
        return TCPTransport(messageType)

    def _onMessage(self, m):
        self._onMessageHandler(m)

    def _onDisconnect(self):
        # Create a completely new TCPTransport
        transport = TCPTransport.createTransport(self._messageType, {})
        # Set the onMessageHandler the same
        transport._onMessageHandler = self._onMessageHandler
        # Note: Leave the onDisconnectHandler as a no-op to prevent recursion
        # Call the user's disconnect handler with the *new* transport
        try:
            self._onDisconnectHandler(transport)
        except Exception, e:
            transport.close()  # Give up
            ERROR("User's onDisconnect handler threw an exception: %s", e)
            #import traceback; traceback.print_exc()
            raise e

        # Here is the "Indiana Jones" moment, where we swap in the connected
        #   socket impl
        self._impl.rob(transport._impl)

    def setOnMessageHandler(self, func):
        self._onMessageHandler = func

    def setOnDisconnectHandler(self, func):
        self._onDisconnectHandler = func

    def connect(self, uri):
        if(type(uri) == str):
            uri = URI(uri)
        self._impl.connect(uri.hostname(), uri.port(), uri.args())

    def close(self):
        self._impl.disconnect()

    def disconnect(self):
        self._impl.disconnect()

    def handleCloseEvent(self):
        self._impl.handleCloseEvent()

    def sendWithoutRetry(self, message):
        m = self._messageStream.serialize(message)
        l = len(m)
        buf = struct.pack("I%ds" % l, socket.htonl(l), m)
        self._impl.send(buf)

    def send(self, message):
        while True:
            try:
                self.sendWithoutRetry(message)
                return
            except Disconnected:
                try:
                    self.handleCloseEvent()
                except RetryOperation:
                    pass

    def allocateMessage(self):
        return self._messageType.allocateMessage()

    def writeQueueSize(self):
        return self._impl.writeQueueSize()

    def readQueueSize(self):
        return self._impl.readQueueSize()

    def flush(self, timeout=None):
        self._impl.flush(timeout)
class RecordStore:
    def __init__(self, completionFunction=lambda k: None):
        "creates a Store"
        self._records = collections.deque()
        self._maxRecord = 0
        self._lock = threading.Lock()
        self._completionFunction = completionFunction

    def store(self, index, key, record):
        "stores record for the associated index and key"
        with self._lock:
            assert(index > self._maxRecord)
            self._records.append((index, key, record))
            self._maxRecord = index

    def discardUpTo(self, index):
        "discards records up to and including the provided index"
        with self._lock:
            while(len(self._records)):
                front = self._records.popleft()
                if(front[0] > index):
                    # Can't discard anymore, let's just return it to the store and exit
                    self._records.appendleft(front)
                    return
                else:
                    # Discarded, let's call the completionFunction
                    try:
                        self._completionFunction(front[1])
                    except Exception, e:
                        ERROR("User completion function threw an exception: %s", e)

    def replay(self, f):
        with self._lock:
            for index, key, record in self._records:
                f(index, key, record)


class Block:
    BLOCK_SIZE = 1024
    HEADER_SIZE = 16
    CAPACITY = BLOCK_SIZE - HEADER_SIZE
    NULL_CONTENT = ""
    NULL_CHILDREN = []
    #   _____________________________________________________________________
    #  | Sequence |  Size  |  Next Record |  CRC32  |        Content         |
    #  |  32bits  | 32bits |    32bits    |  32bits | (BLOCK_SIZE - 16)bytes |
    #   =====================================================================
    #  Sequence    = Sequence Number (for purging/ordering)
    #  Size        = Size of content amongst all blocks in head or data in child
    #  Next Record = Next index, null if tail
    #  Next Record = CRC32, null if continuation, set in head
    #  Content     = The content
    # Rules:
    #    Free Block   = Sequence and Size are 0 (zero).
    #    Head Block   = Sequence and Size are not zero.
    #    Continuation = Sequence is 0, but Size is not.

    def __init__(self, index, block=None):
        if(block):
            assert(len(block) == Block.BLOCK_SIZE)
            s, z, n, c = struct.unpack_from("!IIII", block)
            self._sequence = s
            self._size = z
            self._next = n
            self._crc = c
            # A 'head' block has the size for the entire record, so this will over
            #   shoot in case there's children (which is harmless).
            self._content = block[Block.HEADER_SIZE:
                                  Block.HEADER_SIZE + self._size]
            self._children = []
        else:
            self.clear()
        self._index = index

    def isFree(self):
        return self._sequence == 0 and self._size == 0

    def isHead(self):
        return self._sequence != 0 and self._size != 0

    def isContinuation(self):
        return self._sequence == 0 and self._size != 0

    def getSequence(self):
        return self._sequence

    def setSequence(self, v):
        self._sequence = v

    def getSize(self):
        return self._size

    def setSize(self, v):
        self._size = v

    def getNext(self):
        return self._next

    def setNext(self, v):
        self._next = v

    def getIndex(self):
        return self._index

    def getContent(self):
        if(self.isHead() and len(self.getChildren()) > 0):
            record = [self._content]
            for child in self.getChildren():
                record.append(child.getContent())
            return ''.join(record)
        else:
            return self._content

    def setContent(self, v):
        self._content = v

    def getCRC(self):
        return self._crc

    def setCRC(self, v):
        self._crc = v

    def isCorrupted(self):
        assert(self.isHead())  # should only be called on a head block
        return self._crc != self.computeCRC()

    def serialize(self, memmap):
        s = struct.pack("!IIII%ds" % len(self._content), self._sequence,
                        self._size, self._next, self._crc, self._content)
        memmap.write(s)

    def clear(self):
        self._sequence = 0
        self._size = 0
        self._next = 0
        self._crc = 0
        #self._index   = Never reset the index
        self._content = ''  # Block.NULL_CONTENT
        self._children = []  # Block.NULL_CHILDREN

    def __str__(self):
        if(self.isHead()):
            t = "Head"
        if(self.isContinuation()):
            t = "Continuation"
        if(self.isFree()):
            t = "Free"
        return "<%sBlock index(%d) sequence(%d) size(%d) next(%d) crc(%x) children(%d) content(%s)>" % (t, self._index, self._sequence, self._size, self._next, self._crc, len(self._children), self._content)

    def computeCRC(self):
        crc = binascii.crc32(self._content, 0)
        for child in self._children:
            crc = binascii.crc32(child._content, crc)
        return crc & 0xffffffff

    def getChildren(self):
        return self._children

    def appendChild(self, child):
        assert(self.isHead())
        self._children.append(child)


class FileRecordStore:
    BLOCKS_PER_REGION = 10 * 1024
    BYTES_PER_REGION = BLOCKS_PER_REGION * Block.BLOCK_SIZE
    #   ________________________________________________________
    #  |   Name    |  Version  |  BlocksPerRegion |  BlockSize  |
    #  |  32bytes  |   32bits  |       32bits     |   32bits    |
    #   ========================================================
    #  Name            = 'FileRecordStore'
    #  Version         = Version  (1)
    #  BlocksPerRegion = Number of blocks per region
    #  BlockSize       = Number of bytes per block
    HEADER_SIZE = 44

    def __init__(self, filename, completionFunction=lambda k: None):
        "creates a Store"
        assert(struct.calcsize(
            'IBB') == 6)  # Check if platform assumptions are correct
        self._filename = filename
        self._file = None
        self._freeBlocks = collections.deque()
        self._heads = collections.deque()
        self._regions = []
        self._totalBlocks = 0
        self._maxSequence = 0
        self._lock = threading.Lock()
        self._completionFunction = completionFunction
        self._open()

    def _writeRegionHeader(self, region):
        assert(Block.BLOCK_SIZE >= FileRecordStore.HEADER_SIZE)
        name = 'FileRecordStore'
        version = 1
        s = struct.pack("!32sIII", name + (chr(0) * (32 - len(name))), version,
                        FileRecordStore.BLOCKS_PER_REGION, Block.BLOCK_SIZE)
        region.seek(0)
        region.write(s)

    def _checkRegionHeader(self, region):
        region.seek(0)
        header = region.read(FileRecordStore.HEADER_SIZE)
        name, version, bpr, blocksize = struct.unpack("!32sIII", header)
        if(name.strip(chr(0)) == 'FileRecordStore' and
            version == 1 and
            blocksize == Block.BLOCK_SIZE and
           bpr == FileRecordStore.BLOCKS_PER_REGION):
            return True
        return False

    def _open(self):
        self._file = os.open(self._filename, os.O_CREAT | os.O_RDWR)

        # Grab the file's size and calculate the number of regions.
        #  The nRegions calculation performs integer rounding when the
        #  size is not an even multiple of BYTES_PER_REGION, therefore,
        #  an additional check is made to increment nRegions if there's
        #  a partial region.  The partially written region will be
        #  corrected below -- assuming the header info appears correct.
        stat = os.fstat(self._file)
        nRegions = stat.st_size / FileRecordStore.BYTES_PER_REGION
        if(stat.st_size % FileRecordStore.BYTES_PER_REGION != 0):
            nRegions += 1

        # Create regions for existing file
        #import pdb; pdb.set_trace()
        for n in range(nRegions):
            region = mmap.mmap(self._file, length=FileRecordStore.BYTES_PER_REGION, offset=FileRecordStore.BYTES_PER_REGION * n)
            self._regions.append(region)
            regionHeaderCorrect = self._checkRegionHeader(region)
            if(not regionHeaderCorrect):
                # We abort when the first region isn't correct.
                if(n == 0):
                    raise IOError("File '%s' does not have a correct header indicating that it is a record store." % self._filename)

                # We log and auto-correct all other headers
                WARNING("Region within record store does not have appropriate header, attempting to autocorrect.")
                # This incorrect region is the last of the file and empty, we
                # likely died when trying to allocate and init it properly.
                # Let's just correct it now.
                self._writeRegionHeader(region)
            elif(n == 0):   # header is correct, and we're the first region
                # Check that size is an even multiple of BYTES_PER_REGION
                # If it is not, then we previously died during an append
                # Fix it.
                if(stat.st_size % FileRecordStore.BYTES_PER_REGION != 0):
                    WARNING("File %s is not the appropriate size - autocorrecting it.")
                    os.lseek(self._file, 0, 2)  # the end
                    os.write(self._file, chr(0) * (
                        stat.st_size % FileRecordStore.BYTES_PER_REGION))

        LOG("Initialized %d regions.", len(self._regions))
        self._totalBlocks = nRegions * FileRecordStore.BLOCKS_PER_REGION
        # Extract existing records from file
        currentRecords = {}
        for n in range(self._totalBlocks):
            if(n % FileRecordStore.BLOCKS_PER_REGION != 0):
                # Region header blocks are at the beginning of every region,
                #   we don't add those to the lists.
                block = self._fetchBlock(n)
                if(block.isFree()):
                    # Free Block
                    self._freeBlocks.append(block)
                elif(block.isHead()):
                    # Head blocks need to be added to the map
                    if(block.isCorrupted()):
                        WARNING("Corrupted block found @ %d (sequence = %d)",
                                block.getIndex(), block.getSequence())
                    else:
                        currentRecords[block.getSequence()] = block
        #  append to record list in sequence order for easier discarding
        for sequence in sorted(currentRecords.keys()):
            self._heads.append(currentRecords[sequence])
            self._maxSequence = max(self._maxSequence, sequence)
        LOG("Opened %s: %d records with %d free blocks remaining. Sequences %d through %d." % (self._filename, len(self._heads), len(self._freeBlocks), self.firstStored(), self.lastStored()))

    def _fetchBlock(self, index):
        iRegion = index / FileRecordStore.BLOCKS_PER_REGION
        iOffset = index % FileRecordStore.BLOCKS_PER_REGION
        region = self._regions[iRegion]
        region.seek(iOffset * Block.BLOCK_SIZE)
        block = Block(index, region.read(Block.BLOCK_SIZE))
        if(block.isHead()):
            childIndex = block.getNext()
            while(childIndex != 0):
                child = self._fetchBlock(childIndex)
                block.appendChild(child)
                childIndex = child.getNext()
        return block

    def _writeBlock(self, block):
        index = block.getIndex()
        iRegion = index / FileRecordStore.BLOCKS_PER_REGION
        iOffset = index % FileRecordStore.BLOCKS_PER_REGION
        region = self._regions[iRegion]
        region.seek(iOffset * Block.BLOCK_SIZE)
        block.serialize(region)
        # Now do all the children
        for child in block.getChildren():
            self._writeBlock(child)

    def flush(self):
        with self._lock:
            fstart = time.time()
            for region in self._regions:
                region.flush()
            fend = time.time()
            STATS("elapsed time for flush: %.2f", fend - fstart)

    def store(self, sequence, key, data):
        "stores record for the associated index and key"
        assert(len(key) < 256)
        with self._lock:
            if(sequence <= self._maxSequence):
                WARNING("message discarded because of backtracking: seq(%d) max(%d)", sequence, self._maxSequence)
                return
            packedData = struct.pack(
                "B%ds%ds" % (len(key), len(data)), len(key), key, data)
            block = self._allocateBlock(sequence, packedData)
            block.setCRC(block.computeCRC())
            self._heads.append(block)
            self._writeBlock(block)
            self._maxSequence = max(sequence, self._maxSequence)

    def _allocateBlock(self, sequence, data):
        # First, calculate if we have enough free blocks to service the request
        size = len(data)
        blocksNeeded = max(
            1, (size - 1) / (Block.BLOCK_SIZE - Block.HEADER_SIZE) + 1)
        while(blocksNeeded > len(self._freeBlocks)):
            # Need to allocate more blocks
            dstart = time.time()
            LOG("Allocating another file region (currently %d regions)",
                len(self._regions))
            os.lseek(self._file, 0, 2)  # the end
            os.write(self._file, chr(0) * FileRecordStore.BYTES_PER_REGION)
            newRegion = mmap.mmap(self._file, length=FileRecordStore.BYTES_PER_REGION, offset=FileRecordStore.BYTES_PER_REGION * len(self._regions))
            self._writeRegionHeader(newRegion)
            self._regions.append(newRegion)
            for n in range(1, FileRecordStore.BLOCKS_PER_REGION):
                newRegion.seek(n * Block.BLOCK_SIZE)
                self._freeBlocks.append(Block(self._totalBlocks + n))
            self._totalBlocks += FileRecordStore.BLOCKS_PER_REGION
            LOG("Finished allocating file region (now %d regions)",
                len(self._regions))
            dend = time.time()
            STATS("Allocated %d more blocks in %.2f seconds",
                  FileRecordStore.BLOCKS_PER_REGION, dend - dstart)
        # Next, we setup the "head" block
        head = self._freeBlocks.popleft()
        head.setSequence(sequence)
        head.setSize(size)
        head.setContent(data[0:Block.CAPACITY])
        datai = Block.CAPACITY
        previousBlock = head
        # If the head isn't large enough to store the entire record, then
        #  we need to attach additional children to store the record.
        for n in range(blocksNeeded - 1):
            child = self._freeBlocks.popleft()
            child.setSize(min(Block.CAPACITY, size - datai))
            child.setContent(data[datai:datai + Block.CAPACITY])
            datai += Block.CAPACITY
            previousBlock.setNext(child.getIndex())
            head.appendChild(child)
            previousBlock = child
        return head

    def discardUpTo(self, sequence):
        "discards records up to and including the provided sequence"
        with self._lock:
            dstart = time.time()
            discardCount = 0
            while(len(self._heads)):
                head = self._heads.popleft()
                if(head.getSequence() > sequence):
                    # Can't discard anymore, let's just return it to the store and exit
                    self._heads.appendleft(head)
                    break
                else:
                    # Discarding it
                    data = head.getContent()
                    key = struct.unpack_from('B', data, 0)
                    key = struct.unpack_from('%ds' % (key), data, 1)[0]
                    self._completionFunction(key)
                # Need to clear it out
                discardCount += 1
                for child in head.getChildren():
                    self._freeBlock(child)
                self._freeBlock(head)
            dend = time.time()
            STATS("Discarded %d records in %.2f seconds",
                  discardCount, dend - dstart)

    def _freeBlock(self, block):
        block.clear()
        self._writeBlock(block)
        self._freeBlocks.append(block)

    def replay(self, f):
        with self._lock:
            for block in self._heads:
                data = block.getContent()
                key = struct.unpack_from('B', data, 0)[0]
                key, data = struct.unpack_from(
                    '%ds%ds' % (key, len(data) - key - 1), data, 1)
                f(block.getSequence(), key, data)

    def firstStored(self):
        with self._lock:
            # this shouldn't be called often
            try:
                block = self._heads.popleft()
                sequence = block.getSequence()
                self._heads.appendleft(block)
                return sequence
            except:
                return 0

    def lastStored(self):
        with self._lock:
            return self._maxSequence

    def sanityCheck(self):
        # Count all of the used blocks
        with self._lock:
            totalUsed = 0
            for head in self._heads:
                totalUsed += 1 + len(head.getChildren())
            if(totalUsed + len(self._freeBlocks) + len(self._regions) != self._totalBlocks):
                LOG("%d orphaned blocks, likely due to shutdown during a write." % (self._totalBlocks - (totalUsed + len(self._freeBlocks))))


class BookmarkStore:
    BLOCKS_PER_REGION = 64
    BYTES_PER_REGION = BLOCKS_PER_REGION * Block.BLOCK_SIZE
    HEADER_SIZE = 44
    #   ________________________________________________________
    #  |   Name    |  Version  |  BlocksPerRegion |  BlockSize  |
    #  |  32bytes  |   32bits  |       32bits     |   32bits    |
    #   ========================================================
    #  Name            = 'BookmarkStore'
    #  Version         = Version  (1)
    #  BlocksPerRegion = Number of blocks per region
    #  BlockSize       = Number of bytes per block
    #
    #  Within each Block, we store a key and bookmark each reserved
    #    to 256-bytes and NULL byte padded to take the full 256
    #    bytes.
    #   ___________________________
    #  |    Key     |   Bookmark   |
    #  |  256bytes  |   256bytes   |
    #   ===========================

    def __init__(self, filename):
        "creates a Store"
        assert(Block.CAPACITY > 512)
        self._filename = filename
        self._file = None
        self._freeBlocks = collections.deque()
        self._regions = []
        self._totalBlocks = 0
        self._idMap = {}
        self._lock = threading.Lock()
        self._open()

    def _writeRegionHeader(self, region):
        assert(Block.BLOCK_SIZE >= BookmarkStore.HEADER_SIZE)
        name = 'BookmarkStore'
        version = 1
        s = struct.pack("!32sIII", name + (chr(0) * (32 - len(name))), version,
                        BookmarkStore.BLOCKS_PER_REGION, Block.BLOCK_SIZE)
        region.seek(0)
        region.write(s)

    def _checkRegionHeader(self, region):
        region.seek(0)
        header = region.read(BookmarkStore.HEADER_SIZE)
        name, version, bpr, blocksize = struct.unpack("!32sIII", header)
        if(name.strip(chr(0)) == 'BookmarkStore' and
            version == 1 and
            blocksize == Block.BLOCK_SIZE and
           bpr == BookmarkStore.BLOCKS_PER_REGION):
            return True
        return False

    def _open(self):
        self._file = os.open(self._filename, os.O_CREAT | os.O_RDWR)

        # Grab the file's size and calculate the number of regions.
        #  The nRegions calculation performs integer rounding when the
        #  size is not an even multiple of BYTES_PER_REGION, therefore,
        #  an additional check is made to increment nRegions if there's
        #  a partial region.  The partially written region will be
        #  corrected below -- assuming the header info appears correct.
        stat = os.fstat(self._file)
        nRegions = stat.st_size / BookmarkStore.BYTES_PER_REGION
        if(stat.st_size % BookmarkStore.BYTES_PER_REGION != 0):
            nRegions += 1

        # Create regions for existing file
        #import pdb; pdb.set_trace()
        for n in range(nRegions):
            region = mmap.mmap(self._file, length=BookmarkStore.BYTES_PER_REGION, offset=BookmarkStore.BYTES_PER_REGION * n)
            self._regions.append(region)
            regionHeaderCorrect = self._checkRegionHeader(region)
            if(not regionHeaderCorrect):
                # We abort when the first region isn't correct.
                if(n == 0):
                    raise IOError("File '%s' does not have a correct header indicating that it is a record store." % self._filename)

                # We log and auto-correct all other headers
                WARNING("Region within record store does not have appropriate header, attempting to autocorrect.")
                # This incorrect region is the last of the file and empty, we
                # likely died when trying to allocate and init it properly.
                # Let's just correct it now.
                self._writeRegionHeader(region)
            elif(n == 0):   # header is correct, and we're the first region
                # Check that size is an even multiple of BYTES_PER_REGION
                # If it is not, then we previously died during an append
                # Fix it.
                if(stat.st_size % BookmarkStore.BYTES_PER_REGION != 0):
                    WARNING("File %s is not the appropriate size - autocorrecting it.")
                    os.lseek(self._file, 0, 2)  # the end
                    os.write(self._file, chr(
                        0) * (stat.st_size % BookmarkStore.BYTES_PER_REGION))

        LOG("Initialized %d regions.", len(self._regions))
        self._totalBlocks = nRegions * BookmarkStore.BLOCKS_PER_REGION
        # Extract existing records from file
        for n in range(self._totalBlocks):
            if(n % BookmarkStore.BLOCKS_PER_REGION != 0):
                # Region header blocks are at the beginning of every region,
                #   we don't add those to the lists.
                block = self._fetchBlock(n)
                if(block.isFree()):
                    # Free Block
                    self._freeBlocks.append(block)
                elif(block.isHead()):
                    # Head blocks need to be added to the map
                    if(block.isCorrupted()):
                        WARNING("Corrupted block found @ %d", block.getIndex())
                    else:
                        key = block.getContent()[0:256].strip(chr(0))
                        val = block.getContent()[256:512].strip(chr(0))
                        LOG("Recovered key %s with value %s @ %d" %
                            (key, val, block.getIndex()))
                        self._idMap[key] = block
        LOG("Opened %s: %d keys with %d free blocks remaining." % (
            self._filename, len(self._idMap), len(self._freeBlocks)))

    def _fetchBlock(self, index):
        iRegion = index / BookmarkStore.BLOCKS_PER_REGION
        iOffset = index % BookmarkStore.BLOCKS_PER_REGION
        region = self._regions[iRegion]
        region.seek(iOffset * Block.BLOCK_SIZE)
        block = Block(index, region.read(Block.BLOCK_SIZE))
        if(block.isHead()):
            childIndex = block.getNext()
            while(childIndex != 0):
                child = self._fetchBlock(childIndex)
                block.appendChild(child)
                childIndex = child.getNext()
        return block

    def _writeBlock(self, block):
        index = block.getIndex()
        iRegion = index / BookmarkStore.BLOCKS_PER_REGION
        iOffset = index % BookmarkStore.BLOCKS_PER_REGION
        region = self._regions[iRegion]
        region.seek(iOffset * Block.BLOCK_SIZE)
        block.serialize(region)
        # Now do all the children
        for child in block.getChildren():
            self._writeBlock(child)

    def flush(self):
        with self._lock:
            fstart = time.time()
            for region in self._regions:
                region.flush()
            fend = time.time()
            STATS("elapsed time for flush: %.2f", fend - fstart)

    def store(self, key, bookmark):
        "stores bookmark for the associated key"
        assert(len(key) < 256 and len(bookmark) < 256)
        with self._lock:
            if(key in self._idMap):
                # Already have this value, let's just update
                block = self._idMap[key]
                if(block.isFree() or not block.isHead() or block.isCorrupted()):
                    # Something is wrong, let's just store it in another 'safe'
                    #   location.
                    WARNING("corrupted record found while storing bookmark for %s @ %d, autocorrecting" % (key, block.getIndex()))
                    block = self._allocateBlock("%s%s%s%s" % (key, chr(0) * (256 - len(key)), bookmark, chr(0) * (256 - len(bookmark))))
                    self._idMap[key] = block
                elif(block.isHead()):
                    block.setContent("%s%s%s%s" % (key, chr(0) * (256 - len(
                        key)), bookmark, chr(0) * (256 - len(bookmark))))
            else:
                block = self._allocateBlock("%s%s%s%s" % (key, chr(0) * (256 - len(key)), bookmark, chr(0) * (256 - len(bookmark))))
                self._idMap[key] = block
            block.setCRC(block.computeCRC())
            self._writeBlock(block)

    def get(self, key):
        "retrieves bookmark for the associated key, returns '0' when not found."
        assert(len(key) < 256)
        with self._lock:
            if(key in self._idMap):
                # Already have this value, let's just update
                block = self._idMap[key]
                if(block.isFree() or not block.isHead() or block.isCorrupted()):
                    # Something is wrong, let's just store it in another 'safe'
                    #   location.
                    raise CorruptedRecord("corrupted record for key %s @ %d" %
                                          (key, block.getIndex()))
                elif(block.isHead()):
                    key = block.getContent()[0:256].strip(chr(0))
                    val = block.getContent()[256:512].strip(chr(0))
                    LOG("Recovered key %s with value %s @ %d" %
                        (key, val, block.getIndex()))
                    return val
            else:
                LOG("Could not find key %s, returning '0'." % (key))
                return '0'

    def _allocateBlock(self, data):
        # First, calculate if we have enough free blocks to service the request
        size = len(data)
        blocksNeeded = max(
            1, (size - 1) / (Block.BLOCK_SIZE - Block.HEADER_SIZE) + 1)
        while(blocksNeeded > len(self._freeBlocks)):
            # Need to allocate more blocks
            dstart = time.time()
            LOG("Allocating another file region (currently %d regions)",
                len(self._regions))
            os.lseek(self._file, 0, 2)  # the end
            os.write(self._file, chr(0) * BookmarkStore.BYTES_PER_REGION)
            newRegion = mmap.mmap(self._file, length=BookmarkStore.BYTES_PER_REGION, offset=BookmarkStore.BYTES_PER_REGION * len(self._regions))
            self._writeRegionHeader(newRegion)
            self._regions.append(newRegion)
            for n in range(1, BookmarkStore.BLOCKS_PER_REGION):
                newRegion.seek(n * Block.BLOCK_SIZE)
                self._freeBlocks.append(Block(self._totalBlocks + n))
            self._totalBlocks += BookmarkStore.BLOCKS_PER_REGION
            LOG("Finished allocating file region (now %d regions)",
                len(self._regions))
            dend = time.time()
            STATS("Allocated %d more blocks in %.2f seconds",
                  BookmarkStore.BLOCKS_PER_REGION, dend - dstart)
        # Next, we setup the "head" block
        head = self._freeBlocks.popleft()
        head.setSequence(head.getIndex())
        head.setSize(size)
        head.setContent(data[0:Block.CAPACITY])
        datai = Block.CAPACITY
        previousBlock = head
        # If the head isn't large enough to store the entire record, then
        #  we need to attach additional children to store the record.
        for n in range(blocksNeeded - 1):
            child = self._freeBlocks.popleft()
            child.setSize(min(Block.CAPACITY, size - datai))
            child.setContent(data[datai:datai + Block.CAPACITY])
            datai += Block.CAPACITY
            previousBlock.setNext(child.getIndex())
            head.appendChild(child)
            previousBlock = child
        return head


class Client:
    _NULL_HANDLER = lambda m: None

    def __init__(self, name, transport=None, store=None):
        assert(struct.calcsize(
            'IBB') == 6)  # Check if platform assumptions are correct
        self._name = name
        self._transport = transport
        self._message = None
        if(self._transport):
            self._message = transport.allocateMessage()
        self._haSequenceNumber = 1
        self._publishStore = store
        # Connectivity params
        self._uri = None
        self._handlers = {}
        # For disconnect handling
        self._disconnectHandler = lambda c: None
        # For synchronous ack handling
        self._ackReceived = threading.Condition()
        self._ackQueue = collections.deque()
        self._ackUsername = None
        self._ackPassword = None
        self._awaitingAck = None
        self._exceptionListener = None
        self._username = None
        self._ackReason = None

    def name(self):
        return self._name

    def getName(self):
        return self._name

    def connect(self, uri):
        self._uri = URI(uri)
        if(not self._transport):
            messageType = MessageTypeFactory.createMessageType(
                self._uri.messageType(), self._uri.args())
            self._transport = TransportFactory.createTransport(
                self._uri.transport(), messageType, self._uri.args())
            self._message = self._transport.allocateMessage()
            self._transport.setOnMessageHandler(self._onMessage)
            self._transport.setOnDisconnectHandler(self._onDisconnect)

        self._transport.connect(uri)

    def URI(self):
        return self._uri

    def getURI(self):
        return self._uri

    def setOnDisconnectHandler(self, handler):
        self._disconnectHandler = handler

    def disconnect(self):
        currentTransport = None

        with self._ackReceived:
            currentTransport = self._transport

        if currentTransport:
            currentTransport.disconnect()

        with self._ackReceived:
            self._handlers = {}
            self._ackQueue = collections.deque()
            self._awaitingAck = None
            self._ackReceived.notifyAll()

    def close(self):
        self.disconnect()

    def allocateMessage(self):
        return self._transport.allocateMessage()

    def _sendInternal(self, message):
        while True:
            currentTransport = self._transport
            try:
                currentTransport.sendWithoutRetry(message)
                return
            except Disconnected:
                self._ackReceived.release()
                try:
                    currentTransport.handleCloseEvent()
                except RetryOperation:
                    pass
                finally:
                    self._ackReceived.acquire()

    def send(self, message, onMessage=None, timeout=None):
        with self._ackReceived:
            # short-circut and just send the message if this is invoked
            # like the old 2-argument send()
            if onMessage is None:
                self._transport.send(message)
                return

            command = message.getCommand()

            # For subs and SOWs, we need to add a handler and perform
            # the operation synchronously.

            # I use an in (tuple) here because it's faster than
            # command=='foo' or command=='bar' or ...
            if command in (Command.Subscribe, Command.DeltaSubscribe,
                           Command.SOWAndSubscribe, Command.SOWAndDeltaSubscribe,
                           Command.SOW):
                commandId = self._nextIdentifier()
                message.setCommandId(commandId)
                if command != Command.SOW:
                    message.setSubId(commandId)
                    message.setSendMatchingIds(True)
                message.setQueryId(commandId)
                if message.getAckType() is None:
                    message.setAckType(AckType.Processed)
                else:
                    message.setAckType(
                        AckType.Processed | message.getAckType())
                self._handlers[commandId] = onMessage
                self._syncAckProcessing(message, timeout)
                return commandId
            elif command in (Command.GroupBegin, Command.GroupEnd,
                             Command.OOF, Command.Ack, Command.Unknown):
                # These command types should only be received, not sent.
                raise CommandTypeError(command)
            else:
                if message.getAckType() != AckType.NoAck:
                    commandId = self._nextIdentifier()
                    message.setCommandId(commandId)
                self._transport.send(message)

    # A durable HA publish must have a completionKey defined
    def publish(self, topic, data, completionKey=None):
        with self._ackReceived:
            if(not self._publishStore or not completionKey):
                m = self._message
                m.reset()
                m.setCommand(Command.Publish)
                m.setTopic(topic)
                m.setData(data)
                self._sendInternal(m)
            else:
                m = self._message
                record = struct.pack(
                    'IBB%ds%ds%ds' % (len(data), 7, len(topic)),
                    len(data), 7, len(topic),
                    data, 'publish', topic)
                self._publishStore.store(
                    self._haSequenceNumber, completionKey, record)
                m.reset()
                m.setCommand(Command.Publish)
                m.setTopic(topic)
                m.setAckType(AckType.Persisted)
                m.setData(data)
                m.setSequence(self._haSequenceNumber)
                self._haSequenceNumber += 1
                self._sendInternal(m)

    def deltaPublish(self, topic, data, completionKey=None):
        with self._ackReceived:
            if(not self._publishStore or not completionKey):
                m = self._message
                m.reset()
                m.setCommand(Command.DeltaPublish)
                m.setTopic(topic)
                m.setData(data)
                self._sendInternal(m)
            else:
                m = self._message
                record = struct.pack(
                    'IBB%ds%ds%ds' % (len(data), 13, len(topic)),
                    len(data), 13, len(topic),
                    data, 'delta_publish', topic)
                self._publishStore.store(
                    self._haSequenceNumber, completionKey, record)
                m.reset()
                m.setCommand(Command.DeltaPublish)
                m.setTopic(topic)
                m.setAckType(AckType.Persisted)
                m.setData(data)
                m.setSequence(self._haSequenceNumber)
                self._haSequenceNumber += 1
                self._sendInternal(m)

    def _replayHelper(self, index, key, record):
        data, command, topic = struct.unpack_from('IBB', record, 0)
        data, command, topic = struct.unpack_from(
            '%ds%ds%ds' % (data, command, topic), record, 6)
        # TODO: This could be restructured to not have to reset every time through
        m = self._message
        m.reset()
        m.setAckType(AckType.Persisted)
        m.setCommand(command)
        m.setTopic(topic)
        m.setSequence(index)
        m.setData(data)
        self._haSequenceNumber = index + 1
        self._sendInternal(m)

    def logon(self, timeout=None, authenticator=DefaultAuthenticator()):
        if authenticator is None:
            authenticator = DefaultAuthenticator()

        with self._ackReceived:
            commandId = self._nextIdentifier()
            try:
                self._message.reset()
                self._message.setCommand(Command.Logon)
                self._message.setCommandId(commandId)
                self._message.setClientName(self._name)
                self._message.setAckType(AckType.Processed)
                if self._uri.username() is not None:
                    self._username = self._uri.username()
                    self._message.setUserId(self._username)
                    self._message.setPassword(
                        authenticator.authenticate(self._username,
                                                   self._uri.password()))
                loggedOn = False
                while not loggedOn:
                    try:
                        self._syncAckProcessing(self._message, timeout)
                        loggedOn = True
                    except RetryOperation:
                        self._username = self._ackUsername
                        self._message.setUserId(self. _ackUsername)
                        self._message.setPassword(authenticator.retry(
                                                  self._ackUsername, self._ackPassword))
                authenticator.completed(self._ackUsername, self._ackPassword, self._ackReason)
                # Now, replay all of the data from the store if available
                if(self._publishStore):
                    LOG("starting replay from file for client %s", self._name)
                    self._publishStore.replay(self._replayHelper)
                    LOG("finished replay from file for client %s", self._name)
                return commandId
            except CommandTimedOut, e:
                raise e  # reraise exception

    def subscribe(self, onMessage, topic, filter=None, bookmark=None, timeout=None):
        with self._ackReceived:
            commandId = self._nextIdentifier()
            try:
                self._message.reset()
                self._message.setCommand(Command.Subscribe)
                self._message.setCommandId(commandId)
                self._message.setAckType(AckType.Processed)
                self._message.setTopic(topic)
                self._message.setFilter(filter)
                self._message.setSubId(commandId)
                if(bookmark):
                    self._message.setBookmark(bookmark)
                self._message.setSendMatchingIds(True)
                # Add our handler to the handler map
                self._handlers[commandId] = onMessage
                self._syncAckProcessing(self._message, timeout)
                return commandId
            except CommandTimedOut, e:
                self.unsubscribe(commandId)
                raise e  # reraise exception

    def deltaSubscribe(self, onMessage, topic, filter=None, timeout=None):
        with self._ackReceived:
            commandId = self._nextIdentifier()
            try:
                self._message.reset()
                self._message.setCommand(Command.DeltaSubscribe)
                self._message.setCommandId(commandId)
                self._message.setAckType(AckType.Processed)
                self._message.setTopic(topic)
                self._message.setFilter(filter)
                self._message.setSubId(commandId)
                self._message.setSendMatchingIds(True)
                # Add our handler to the handler map
                self._handlers[commandId] = onMessage
                self._syncAckProcessing(self._message, timeout)
                return commandId
            except CommandTimedOut, e:
                self.unsubscribe(commandId)
                raise e  # reraise exception

    ## Unsubscribes this Client from a single subscription,
    #  or all of the subscriptions on self.
    #
    #  @param subId The subscription ID to unsubscribe from, as returned
    #     from a call to subscribe() or a variant.
    def unsubscribe(self, subId=None):
        if(not subId):
            subId = "all"
        with self._ackReceived:
            commandId = self._nextIdentifier()
            self._message.reset()
            self._message.setCommand(Command.Unsubscribe)
            self._message.setCommandId(commandId)
            self._message.setSubId(subId)
            if subId in self._handlers:
                del self._handlers[subId]
            elif subId != "all":
                return
            self._sendInternal(self._message)

    def sow(self, onMessage, topic, filter, batchSize=1, timeout=None):
        with self._ackReceived:
            commandId = self._nextIdentifier()
            try:
                self._message.reset()
                self._message.setCommand(Command.SOW)
                self._message.setCommandId(commandId)
                self._message.setQueryId(commandId)
                self._message.setAckType(AckType.Processed)
                self._message.setTopic(topic)
                self._message.setFilter(filter)
                self._message.setBatchSize(batchSize)
                # Add our handler to the handler map
                self._handlers[commandId] = onMessage
                self._syncAckProcessing(self._message, timeout)
                return commandId
            except CommandTimedOut, e:
                del self._handlers[commandId]
                raise e  # reraise exception

    def sowAndSubscribe(self, onMessage, topic, filter, batchSize=1, timeout=None):
        with self._ackReceived:
            commandId = self._nextIdentifier()
            try:
                self._message.reset()
                self._message.setCommand(Command.SOWAndSubscribe)
                self._message.setCommandId(commandId)
                self._message.setSubId(commandId)
                self._message.setQueryId(commandId)
                self._message.setAckType(AckType.Processed)
                self._message.setTopic(topic)
                self._message.setFilter(filter)
                self._message.setBatchSize(batchSize)
                # Add our handler to the handler map
                self._handlers[commandId] = onMessage
                self._syncAckProcessing(self._message, timeout)
                return commandId
            except CommandTimedOut, e:
                self.unsubscribe(commandId)
                raise e  # reraise exception

    def sowDelete(self, onMessage, topic, filter, timeout=None):
        with self._ackReceived:
            commandId = self._nextIdentifier()
            try:
                self._message.reset()
                self._message.setCommand(Command.SOWDelete)
                self._message.setCommandId(commandId)
                self._message.setSubId(commandId)
                self._message.setQueryId(commandId)
                self._message.setAckType(AckType.Processed | AckType.Stats)
                self._message.setTopic(topic)
                self._message.setFilter(filter)
                self._handlers[commandId] = onMessage
                self._syncAckProcessing(self._message, timeout)
                return commandId
            except CommandTimedOut, e:
                del self._handlers[commandId]
                raise e

    def sowAndDeltaSubscribe(self, onMessage, topic, filter, batchSize=1, timeout=None):
        with self._ackReceived:
            commandId = self._nextIdentifier()
            try:
                self._message.reset()
                self._message.setCommand(Command.SOWAndDeltaSubscribe)
                self._message.setCommandId(commandId)
                self._message.setSubId(commandId)
                self._message.setQueryId(commandId)
                self._message.setAckType(AckType.Processed)
                self._message.setTopic(topic)
                self._message.setFilter(filter)
                self._message.setBatchSize(batchSize)
                # Add our handler to the handler map
                self._handlers[commandId] = onMessage
                self._syncAckProcessing(self._message, timeout)
                return commandId
            except CommandTimedOut, e:
                self.unsubscribe(commandId)
                raise e  # reraise exception

    def flush(self):
        self._transport.flush()

    def _nextIdentifier(self):
        return base64.b64encode(uuid.uuid4().bytes)[:-2]

    def _nullHandler(self, message):
        pass

    # This is a helper function used to encapsulate the logic for
    #   waiting so that the async commands like 'subscribe' and 'sow'
    #   can have an easier to use exception interface for failed
    #   command execution.
    def _syncAckProcessing(self, message, timeout):
        # Here's the boilerplate for waiting for the processed ack and
        #   raising exceptions should they occur.
        commandId = message.getCommandId()
        if(timeout > 0):
            # Convert from milliseconds to seconds
            timeout = float(timeout) / 1000  
        else:
            timeout = None  # 0 means No timeout

        # Notify the client onMessage func that we're the one waiting
        #   for the ack.
        self._awaitingAck = commandId
        # Send the command
        self._sendInternal(message)
        # Wait until the ack is received, or time out occurs
        # Note: If we hang here, it's likely because the 'processed' ack
        #       wasn't requested on the message.
        self._ackReceived.wait(timeout)
        # We're no longer waiting for an ack
        try:
            ack = self._ackQueue.popleft()
        except IndexError:
            ack = None
        if(ack):
            status, self._ackReason, seq, self._ackUsername, self._ackPassword = ack
            if(status != Status.Failure):
                if(message.getCommand() == Command.Logon):
                    if status == Status.Retry:
                        raise RetryOperation(
                            "authentication module requested a retry.")
                    # If we're a 'logon' command, we need to extract
                    #   the sequence number, if available
                    if(self._publishStore):
                        self._publishStore.discardUpTo(seq)
                    self._haSequenceNumber = seq + 1
                # all is good, just return
                return
            if(commandId in self._handlers):
                del self._handlers[commandId]
            if(self._ackReason == Reason.BadFilter):
                # Whoops, a bad filter, let's raise the BadFilter exception
                raise BadFilter(message.getFilter())
            elif(self._ackReason == Reason.BadRegexTopic):
                # poorly formed regex topic
                raise BadRegexTopic(message.getTopic())
            elif(self._ackReason == Reason.SubscriptionAlreadyExists):
                # sub-id already exists
                raise SubscriptionAlreadyExists(commandId)
            elif(self._ackReason == Reason.NameInUse):
                # client name is already in use
                raise ClientNameInUse(self.name())
            elif(self._ackReason == Reason.AuthFailure):
                # Logon failure, typically
                raise AuthenticationError(self._uri.username())
            elif(self._ackReason == Reason.NotEntitled):
                raise NotEntitledError(self._username, message.getTopic())
            # We're not successful and not a bad filter -- what are we?
            raise UnknownError(self._ackReason)
        else:
            # No ack = timeout
            raise CommandTimedOut(commandId)

    def _onMessage(self, message):
        if(message.getCommand() == Command.Ack):
            if(message.getAckType() == AckType.Persisted):
                # Best Practice:  If you don't care about the dupes that occur during
                #  failover or rapid disconnect/reconnect, then just ignore them.
                #  We could discard each duplicate from the persisted store, but the
                #  storage costs of doing 1 record discards is heavy.  In most scenarios
                #  we'll just quickly blow through the duplicates and get back to
                #  processing the non-dupes.
                if(self._publishStore and not message.getReason() == Reason.Duplicate):
                    try:
                        self._publishStore.discardUpTo(message.getSequence())
                    except:
                        self.absorbedException(sys.exc_info())
            key = message.getCommandId()
            if(key):
                # If this command is awaiting an ack, let's notify it that
                #    it's arrived
                with self._ackReceived:
                    if(self._awaitingAck == key):
                        # Clear the waiting ack
                        self._awaitingAck = None
                        # Queue ack and notify the waiter
                        self._ackQueue.append((message.getStatus(), message.getReason(), message.getSequence(), message.getUserId(), message.getPassword()))
                        self._ackReceived.notifyAll()
                        return
        key = message.getSubIds()
        if(key):
            # Publish messages coming through on a subscription
            #   (Could also be a query result from a sow_and_subscribe)
            for k in key.split(','):
                try:
                    self._handlers.get(k, self._nullHandler)(message)
                except:
                    self.absorbedException(sys.exc_info())
            return
        key = message.getQueryId()
        if(key):
            # Query results coming through on a standard sow query
            try:
                self._handlers.get(key, self._nullHandler)(message)
            except:
                self.absorbedException(sys.exc_info())
            return
        key = message.getCommandId()
        if(key):
            # Query results coming through on a standard sow query
            try:
                self._handlers.get(key, self._nullHandler)(message)
            except:
                self.absorbedException(sys.exc_info())
            return

    def _onDisconnect(self, transport):
        # Swap the transport momentarily for the reconnect attempt
        with self._ackReceived:
            oldTransport = self._transport
            self._transport = transport
            try:
                self._disconnectHandler(self)
            finally:
                self._transport = oldTransport
            with self._ackReceived:
                self._ackReceived.notifyAll()

    ## Sets the handler function invoked when an exception occurs
    #  in a thread and must be absorbed.
    #  @param self The reference to self
    #  @param exceptionListener_ The function object to be called with \c sys.exc_info()
    #         when an exception occurs and is absorbed.
    def setExceptionListener(self, exceptionListener_):
        self._exceptionListener = exceptionListener_

    def absorbedException(self, anException_):
        try:
            self._exceptionListener(anException_)
        except:
            pass
